#include "users_history.h"

#include <passport/infra/daemons/lbchdb/src/yt_converters/common.h>
#include <passport/infra/daemons/lbchdb/src/yt_converters/utils.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/yt/yt_impl.h>

#include <yt/yt/client/table_client/comparator.h>
#include <yt/yt/client/table_client/helpers.h>
#include <yt/yt/client/table_client/logical_type.h>
#include <yt/yt/client/table_client/name_table.h>
#include <yt/yt/client/table_client/row_buffer.h>

#include <util/generic/hash_set.h>

namespace NPassport::NLbchdb::NYtConv::NMailUserJournal {
    NYt::TWriteQuery TUserHistoryQueryConverter::Convert(const TString& table,
                                                         size_t offset,
                                                         size_t count) const {
        return ConvertImpl(table, GetSpan(offset, count));
    }

    static NYT::NTableClient::TNameTablePtr CreateNameTable() {
        NYT::NTableClient::TNameTablePtr res = NYT::New<NYT::NTableClient::TNameTable>();
        res->RegisterName("uid");
        res->RegisterName("reversed_unixtime");
        res->RegisterName("operation");
        res->RegisterName("module");
        res->RegisterName("data");
        return res;
    }
    static const NYT::NTableClient::TNameTablePtr NAME_TABLE = CreateNameTable();

    NYt::TWriteQuery TUserHistoryQueryConverter::ConvertImpl(const TString& table,
                                                             std::span<const TUserHistoryEntry> data) {
        NYt::TWriteQuery res;
        res.Impl = std::make_unique<NYt::TWriteQueryImpl>();
        NYt::TWriteSubQuery& sub = res.Impl->Subqueries.emplace_back();

        sub.Path = table;
        sub.NameTable = NAME_TABLE;

        NYT::NTableClient::TUnversionedRowsBuilder builder;
        builder.ReserveRows(data.size());

        for (const TUserHistoryEntry& d : data) {
            builder.AddRow(
                d.Uid,
                d.ReversedUnixtime,
                d.Operation,
                d.Module,
                d.Data);
        }

        sub.Range = builder.Build();

        return res;
    }

    const TString TUsersHistory::TABLE_NAME = "users_history";
    const TString TUsersHistory::CORP_TABLE_NAME = "corp_users_history";

    TUserHistoryEntry TUsersHistory::BuildEntry(const NParser::TMailUserJournalRow& entry, size_t compressIfMoreThan) {
        return TUserHistoryEntry{
            .Uid = entry.Uid,
            .ReversedUnixtime = TUtils::ReverseTimeT(entry.Date),
            .Operation = TString(entry.GetField("operation")),
            .Module = TString(entry.GetField("module")),
            .Data = BuildData(entry, compressIfMoreThan),
        };
    }

    static const THashSet<TString> FIELDS_TO_STORE = {
        "abuseType",
        "affected",
        "browser.name",
        "browser.version",
        "condition",
        "destFid",
        "deviceType",
        "emailFrom",
        "fid",
        "filterIds",
        "folder_symbol",
        "ftype",
        "hidden",
        "internetProvider",
        "ip",
        "lids",
        "lidText",
        "mid",
        "msgStatus",
        "operationSystem.name",
        "operationSystem.version",
        "regionId",
        "spam_type",
        "state",
        "target",
        "view_date",
    };

    NYT::NYson::TYsonString TUsersHistory::BuildData(const NParser::TMailUserJournalRow& entry,
                                                     size_t compressIfMoreThan) {
        struct TToCompress {
            TStringBuf Key;
            TStringBuf Value;
        };
        TStackVec<TToCompress, 2> toCompress;

        TStringStream stream;
        stream.Reserve(1024);

        NYT::NYson::TBufferedBinaryYsonWriter writer(&stream);
        writer.OnBeginMap();

        for (const auto& [key, val] : entry.Tskv.Fields()) {
            if (val.empty() || !FIELDS_TO_STORE.contains(key)) {
                continue;
            }

            if (val.size() > compressIfMoreThan) {
                toCompress.push_back({.Key = key, .Value = val});
            } else {
                writer.OnKeyedItem(key);
                writer.OnStringScalar(val);
            }
        }

        if (!toCompress.empty()) {
            writer.OnKeyedItem("_compressed");
            writer.OnBeginMap();

            for (const TToCompress& c : toCompress) {
                writer.OnKeyedItem(c.Key);

                writer.OnBeginMap();
                writer.OnKeyedItem("size");
                writer.OnUint64Scalar(c.Value.size());
                writer.OnKeyedItem("value");
                writer.OnStringScalar(TCompressor::Compress(c.Value, TCompressor::ECompressionCodec::Brotli));
                writer.OnEndMap();
            }

            writer.OnEndMap();
        }

        writer.OnEndMap();
        writer.Flush();

        return NYT::NYson::TYsonString(stream.Str());
    }

    NYt::TTableSchema TUsersHistory::CreateYtSchema() {
        auto column = [](const TString& name, NYT::NTableClient::ESimpleLogicalValueType type) {
            return NYT::NTableClient::TColumnSchema(
                name,
                SimpleLogicalType(type),
                NYT::NTableClient::ESortOrder::Ascending);
        };

        NYt::TTableSchema res;
        res.Impl = std::make_shared<NYt::TTableSchemaImpl>();

        res.Impl->Schema = NYT::NTableClient::TTableSchema(
            std::vector<NYT::NTableClient::TColumnSchema>({
                column("uid", NYT::NTableClient::ESimpleLogicalValueType::Uint64),
                column("reversed_unixtime", NYT::NTableClient::ESimpleLogicalValueType::Int64),
                column("operation", NYT::NTableClient::ESimpleLogicalValueType::String),
                column("module", NYT::NTableClient::ESimpleLogicalValueType::String),
                NYT::NTableClient::TColumnSchema(
                    "data",
                    NYT::NTableClient::ESimpleLogicalValueType::Any),
            }),
            true,
            true);

        return res;
    }
}

using namespace NPassport::NLbchdb::NYtConv::NMailUserJournal;

template <>
void Out<TUserHistoryEntry>(IOutputStream& o, const TUserHistoryEntry& value) {
    o << "uid=" << value.Uid
      << ". reversed_unixtime=" << value.ReversedUnixtime
      << ". unixtime=" << NPassport::NLbchdb::NYtConv::TUtils::ReverseTimeT(value.ReversedUnixtime)
      << ". operation=" << value.Operation
      << ". module=" << value.Module
      << ". data=" << value.Data
      << Endl;
}
