#include "mail_user_journal_converter.h"

namespace NPassport::NLbchdb::NYtConv {
    TMailUserJournalConverter::TMailUserJournalConverter() = default;

    void TMailUserJournalConverter::Reserve(size_t size) {
        Reserve_ = size;
    }

    void TMailUserJournalConverter::Process(
        const NParser::TMailUserJournalRow& entry,
        size_t compressIfMoreThan,
        const TTablesLifeTimeConfig& config) {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");

        const bool isCorp = entry.GetField("tableName") == "corp_users_history";
        const TInstant date = TInstant::Seconds(entry.Date / 1000);
        if (!config.IsDateActual(date, isCorp ? config.MailCorpTtl : config.MailProdTtl)) {
            TLog::Warning() << "MailUserJournalConverter: got non-actual date in event: " << date
                            << ". uid=" << entry.Uid
                            << ". operation=" << entry.GetField("operation")
                            << ". module=" << entry.GetField("module");
            return;
        }

        auto& conv = isCorp ? Corp_ : Prod_;

        NMailUserJournal::TUserHistoryQueryConverterPtr& ptr =
            conv.emplace(NYt::TMonthlyTable(date), nullptr).first->second;
        if (!ptr) {
            ptr = std::make_unique<NMailUserJournal::TUserHistoryQueryConverter>();
            ptr->Reserve(Reserve_);
        }

        ptr->Add(NMailUserJournal::TUsersHistory::BuildEntry(entry, compressIfMoreThan));
    }

    NYt::TWriteQueries<TString> TMailUserJournalConverter::Finish() {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");
        IsFinished_ = true;

        NYt::TWriteQueries<TString> res;
        auto add = [&](auto& map, const TString& prefix) {
            for (auto& [key, value] : map) {
                res.ByTable.emplace(
                    NUtils::CreateStr(prefix, "/", key.TableName()),
                    std::move(value));
            }
        };

        add(Corp_, NMailUserJournal::TUsersHistory::CORP_TABLE_NAME);
        add(Prod_, NMailUserJournal::TUsersHistory::TABLE_NAME);

        return res;
    }
}
