#include "push_converter.h"

namespace NPassport::NLbchdb::NYtConv {
    void TPushConverter::TQueryConverters::Init(size_t reserve) {
        Push = std::make_unique<NPush::TPushQueryConverter>();
        Push->Reserve(reserve);

        PushByUid = std::make_unique<NPush::TPushByUidQueryConverter>();
        PushByUid->Reserve(reserve);

        PushByDeviceId = std::make_unique<NPush::TPushByDeviceIdQueryConverter>();
        PushByDeviceId->Reserve(reserve);

        PushByAppId = std::make_unique<NPush::TPushByAppIdQueryConverter>();
        PushByAppId->Reserve(reserve);
    }

    TPushConverter::TPushConverter() = default;

    void TPushConverter::Reserve(size_t size) {
        Reserve_ = size;
    }

    void TPushConverter::Process(const NParser::TPushRow& row, const TTablesLifeTimeConfig& config) {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");

        if (!config.IsDateActual(row.Unixtime, config.PushTtl)) {
            TLog::Warning() << "PushConverter: got non-actual date in event: " << row.Unixtime
                            << ". push_id=" << row.PushId
                            << ". subscription_id=" << row.SubscriptionId;
            return;
        }

        NYt::TMonthlyTable table(row.Unixtime);

        auto it = Tables_.find(table);
        if (it == Tables_.end()) {
            it = Tables_.emplace(table, TQueryConverters{}).first;
            it->second.Init(Reserve_);
        }

        TQueryConverters& converters = it->second;
        NPush::TEntryBuilder entryBuilder(row);

        converters.Push->Add(entryBuilder.BuildPushEntry());
        converters.PushByUid->Add(entryBuilder.BuildPushIndxEntry());
        if (!row.DeviceId.empty()) {
            converters.PushByDeviceId->Add(entryBuilder.BuildPushIndxEntry());
        }
        if (!row.AppId.empty()) {
            converters.PushByAppId->Add(entryBuilder.BuildPushIndxEntry());
        }
    }

    NYt::TWriteQueries<TString> TPushConverter::Finish() {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");
        IsFinished_ = true;

        NYt::TWriteQueries<TString> res;
        for (auto& [key, value] : Tables_) {
            res.ByTable.emplace(
                NUtils::CreateStr(NPush::TPushTable::TABLE_NAME, '/', key.TableName()),
                std::move(value.Push));
            res.ByTable.emplace(
                NUtils::CreateStr(NPush::TPushByUidTable::TABLE_NAME, '/', key.TableName()),
                std::move(value.PushByUid));
            res.ByTable.emplace(
                NUtils::CreateStr(NPush::TPushByDeviceIdTable::TABLE_NAME, '/', key.TableName()),
                std::move(value.PushByDeviceId));
            res.ByTable.emplace(
                NUtils::CreateStr(NPush::TPushByAppIdTable::TABLE_NAME, '/', key.TableName()),
                std::move(value.PushByAppId));
        }

        return res;
    }
}
