#include "restore.h"

#include <passport/infra/daemons/lbchdb/src/yt_converters/common.h>
#include <passport/infra/daemons/lbchdb/src/yt_converters/utils.h>

#include <passport/infra/libs/cpp/yt/yt_impl.h>

#include <yt/yt/client/table_client/helpers.h>
#include <yt/yt/client/table_client/name_table.h>
#include <yt/yt/client/table_client/row_buffer.h>
#include <yt/yt/core/yson/writer.h>

namespace NPassport::NLbchdb::NYtConv::NRestore {
    const TString TRestore::TABLE_NAME = "restore/restore";

    static NYT::NYson::TYsonString BuildData(const NParser::TRestoreRow& row,
                                             const NCrypto::TEncryptor& encryptor,
                                             size_t compressIfMoreThan = 100 * 1024) {
        TStringStream stream;
        stream.Reserve(1024);

        NYT::NYson::TBufferedBinaryYsonWriter writer(&stream);
        writer.OnBeginMap();

        auto writeField = [&writer](const TStringBuf key, const TStringBuf value) {
            writer.OnKeyedItem(key);
            writer.OnStringScalar(value);
        };

        writeField("action", row.Action);
        writeField("restore_id", row.RestoreId);

        writer.OnKeyedItem("data_json");
        TYsonSerializer::SerializeEncrypted(
            writer,
            row.DataJson,
            encryptor,
            NCrypto::TEncryptor::EKeyRing::Restore,
            TEncryptionSettings{
                .CompressIfMoreThan = compressIfMoreThan,
                .Codec = TCompressor::ECompressionCodec::GZip,
            });

        writer.OnEndMap();
        writer.Flush();

        return NYT::NYson::TYsonString(stream.Str());
    }

    TRestoreEntry TRestoreEntry::BuildFrom(
        const NParser::TRestoreRow& row,
        const NCrypto::TEncryptor& encryptor,
        size_t compressIfMoreThan) {
        return TRestoreEntry{
            .Uid = row.Uid,
            .ReversedTimestamp = TUtils::ReverseTimeT(row.Timestamp.MicroSeconds()),
            .Data = BuildData(row, encryptor, compressIfMoreThan),
        };
    }

    NYt::TWriteQuery TRestoreQueryConverter::Convert(const TString& table,
                                                     size_t offset,
                                                     size_t count) const {
        return ConvertImpl(table, GetSpan(offset, count));
    }

    static NYT::NTableClient::TNameTablePtr CreateNameTable() {
        NYT::NTableClient::TNameTablePtr res = NYT::New<NYT::NTableClient::TNameTable>();
        res->RegisterName("uid");
        res->RegisterName("reversed_timestamp");
        res->RegisterName("data");
        return res;
    }
    static const NYT::NTableClient::TNameTablePtr NAME_TABLE = CreateNameTable();

    NYt::TWriteQuery TRestoreQueryConverter::ConvertImpl(const TString& table,
                                                         std::span<const TRestoreEntry> data) {
        NYt::TWriteQuery res;

        res.Impl = std::make_unique<NYt::TWriteQueryImpl>();
        NYt::TWriteSubQuery& sub = res.Impl->Subqueries.emplace_back();

        sub.Path = table;
        sub.NameTable = NAME_TABLE;

        NYT::NTableClient::TUnversionedRowsBuilder builder;
        builder.ReserveRows(data.size());

        for (const auto& entry : data) {
            builder.AddRow(
                entry.Uid,
                entry.ReversedTimestamp,
                entry.Data);
        }

        sub.Range = builder.Build();

        return res;
    }
}

using namespace NPassport::NLbchdb::NYtConv::NRestore;

template <>
void Out<TRestoreEntry>(IOutputStream& o, const TRestoreEntry& value) {
    o << "uid=" << value.Uid
      << ". reversed_timestamp=" << value.ReversedTimestamp
      << ". data=" << value.Data
      << Endl;
}
