#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/auths.h>

#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NHBaseConv::NAuth;

Y_UNIT_TEST_SUITE(ConvertBbAuthAuths) {
    Y_UNIT_TEST(buildKeyEpoch) {
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildKeyEpoch(100500), 0);
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildKeyEpoch(10050000000), 3877);
    }

    Y_UNIT_TEST(buildReverseTimestamp) {
        UNIT_ASSERT_VALUES_EQUAL(
            TAuthQuery::BuildReverseTimestamp(TInstant()),
            "9223372036854775807.000000");
        UNIT_ASSERT_VALUES_EQUAL(
            TAuthQuery::BuildReverseTimestamp(TInstant::ParseIso8601("2010-03-16T11:02:55.123456+03")),
            "9223372035586049232.876544");
        UNIT_ASSERT_VALUES_EQUAL(
            TAuthQuery::BuildReverseTimestamp(TInstant::ParseIso8601("2010-03-16T11:02:55.999999+03")),
            "9223372035586049232.000001");
        UNIT_ASSERT_VALUES_EQUAL(
            TAuthQuery::BuildReverseTimestamp(TInstant::ParseIso8601("2010-03-16T11:02:55.000000+03")),
            "9223372035586049232.000000");
    }

    Y_UNIT_TEST(buildParams) {
        NExtend::TAuthExtendedEntry entry;

        NHbase::TQuery q;
        TAuthQuery::BuildParams(entry, q);

        UNIT_ASSERT_VALUES_EQUAL(
            q,
            NHbase::TQuery({
                .Params = {
                    {"c:host_id", "0"},
                },
            }));
        q.Params.clear();

        entry.Row.ClientName = "bb";
        entry.Row.HostId = 789;
        entry.Row.Login = "vasya";
        entry.Row.Sid = "some sid";
        entry.Row.AuthType = "oauth_kek";
        entry.Row.Status = "some status";
        entry.Row.YandexUid = "some yandexuid";
        entry.Row.Comment = "some comment";
        entry.Row.ProxyIp = "8.8.8.8";
        entry.Row.UserIp = "8.8.8.0";
        entry.IpData = NExtend::TIpData{
            .GeoId = 42,
            .As = "some as",
            .IsYandex = true,
        };
        entry.UaData = NExtend::TUserAgentData{
            .BrowserName = "kek",
            .BrowserVersion = "ver1",
            .OsName = "lubuntu",
            .OsFamily = "lol",
            .OsVersion = "1.0.1e",
        };
        TAuthQuery::BuildParams(entry, q);
        UNIT_ASSERT_VALUES_EQUAL(
            q,
            NHbase::TQuery({
                .Params = {
                    {"c:host_id", "789"},
                    {"c:client_name", "bb"},
                    {"c:login", "vasya"},
                    {"c:sid", "some sid"},
                    {"c:type", "oauth_kek"},
                    {"c:status", "some status"},
                    {"c:yandexuid", "some yandexuid"},
                    {"c:comment", "some comment"},
                    {"c:user_ip", "8.8.8.8"},
                    {"c:ip.geoid", "42"},
                    {"c:ip.as_list", "some as"},
                    {"c:ip.is_yandex", "1"},
                    {"c:browser.name", "kek"},
                    {"c:browser.version", "ver1"},
                    {"c:os.name", "lubuntu"},
                    {"c:os.family", "lol"},
                    {"c:os.version", "1.0.1e"},
                },
            }));
        q.Params.clear();

        entry.Row.ProxyIp.Clear();
        entry.IpData->IsYandex = false;
        TAuthQuery::BuildParams(entry, q);
        UNIT_ASSERT_VALUES_EQUAL(
            q,
            NHbase::TQuery({
                .Params = {
                    {"c:host_id", "789"},
                    {"c:client_name", "bb"},
                    {"c:login", "vasya"},
                    {"c:sid", "some sid"},
                    {"c:type", "oauth_kek"},
                    {"c:status", "some status"},
                    {"c:yandexuid", "some yandexuid"},
                    {"c:comment", "some comment"},
                    {"c:user_ip", "8.8.8.0"},
                    {"c:ip.geoid", "42"},
                    {"c:ip.as_list", "some as"},
                    {"c:browser.name", "kek"},
                    {"c:browser.version", "ver1"},
                    {"c:os.name", "lubuntu"},
                    {"c:os.family", "lol"},
                    {"c:os.version", "1.0.1e"},
                },
            }));
        q.Params.clear();
    }

    Y_UNIT_TEST(buildRow) {
        NExtend::TAuthExtendedEntry entry;
        entry.Row.Timestamp = TInstant::ParseIso8601("2010-03-16T11:02:55.123456+03");
        entry.Row.AuthType = "oauth_kek";
        entry.Row.Uid = 789;
        entry.Row.HostId = 127;

        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildRow(entry, false),
                                 "489_789_9223372035586049232.876544_127");
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildRow(entry, true),
                                 "489_789_9223372035586049232.876544_oauth_kek/-");

        entry.Row.AuthType = "oauthcheck";
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildRow(entry, false),
                                 "489_789_9223372035586049232.876544_127");
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildRow(entry, true),
                                 "489_789_9223372035586049232.876544_oauthcheck/-");

        entry.Comment = NExtend::TComment::Create("tokid=value1;");
        UNIT_ASSERT_VALUES_EQUAL(TAuthQuery::BuildRow(entry, true),
                                 "489_789_9223372035586049232.876544_oauthcheck/value1");
    }
}

template <>
void Out<NHbase::TQuery>(IOutputStream& o, const NHbase::TQuery& value) {
    o << value.PrintDebug();
}
