#include <passport/infra/daemons/lbchdb/ut/common.h>

#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/succed_auths.h>
#include <passport/infra/daemons/lbchdb/src/proto/auths.pb.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NHBaseConv::NAuth;

Y_UNIT_TEST_SUITE(ConvertBbAuthSuccedAuths) {
    Y_UNIT_TEST(shouldSkip) {
        NExtend::TAuthExtendedEntry entry;

        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "kek";
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "kek";
        entry.Row.Status = "lol";
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "kek";
        entry.Row.Status = "successful";
        UNIT_ASSERT(!TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "web";
        entry.Row.Status = "successful";
        UNIT_ASSERT(!TSuccessAuth::ShouldSkip(entry));

        entry.Row.ClientName = "bb";
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "web";
        entry.Row.Status = "ses_create";
        UNIT_ASSERT(!TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "unknown";
        entry.Row.Status = "ses_create";
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "oauthcreate";
        entry.Row.Status = "ses_create";
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "kek";
        entry.Row.Status = "successful";
        entry.Comment = NExtend::TComment::Create("asrc=kek;");
        UNIT_ASSERT(!TSuccessAuth::ShouldSkip(entry));

        entry.Row.AuthType = "kek";
        entry.Row.Status = "successful";
        entry.Comment = NExtend::TComment::Create("asrc=xtoken;");
        UNIT_ASSERT(TSuccessAuth::ShouldSkip(entry));
    }

    Y_UNIT_TEST(fillBrowserInfo) {
        NExtend::TAuthExtendedEntry entry;
        AuthInfo proto;

        TSuccessAuth::FillBrowserInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        entry.UaData = NExtend::TUserAgentData{};
        TSuccessAuth::FillBrowserInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        entry.UaData = NExtend::TUserAgentData{.BrowserName = "kek"};
        TSuccessAuth::FillBrowserInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "browser_info {\n  name: \"kek\"\n}\n");

        proto = {};
        entry.UaData = NExtend::TUserAgentData{.BrowserVersion = "ver1"};
        TSuccessAuth::FillBrowserInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "browser_info {\n  version: \"ver1\"\n}\n");

        proto = {};
        entry.UaData = NExtend::TUserAgentData{
            .BrowserName = "kek",
            .BrowserVersion = "ver1",
        };
        TSuccessAuth::FillBrowserInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "browser_info {\n  name: \"kek\"\n  version: \"ver1\"\n}\n");
    }

    Y_UNIT_TEST(fillOsInfo) {
        NExtend::TAuthExtendedEntry entry;
        AuthInfo proto;

        TSuccessAuth::FillOsInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        entry.UaData = NExtend::TUserAgentData{};
        TSuccessAuth::FillOsInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        entry.UaData = NExtend::TUserAgentData{.OsName = "lubuntu"};
        TSuccessAuth::FillOsInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "os_info {\n  name: \"lubuntu\"\n}\n");

        proto = {};
        entry.UaData = NExtend::TUserAgentData{.OsVersion = "1.0.1e"};
        TSuccessAuth::FillOsInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "os_info {\n  version: \"1.0.1e\"\n}\n");

        proto = {};
        entry.UaData = NExtend::TUserAgentData{
            .OsName = "lubuntu",
            .OsVersion = "1.0.1e",
        };
        TSuccessAuth::FillOsInfo(entry, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "os_info {\n  name: \"lubuntu\"\n  version: \"1.0.1e\"\n}\n");
    }

    Y_UNIT_TEST(fillTokenInfo) {
        NExtend::TComment comment;
        AuthInfo proto;

        proto = {};
        comment = NExtend::TComment::Create("");
        TSuccessAuth::FillTokenInfo(comment, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        proto = {};
        comment = NExtend::TComment::Create("tokid=token_id;");
        TSuccessAuth::FillTokenInfo(comment, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        proto = {};
        comment = NExtend::TComment::Create("clid=client_id;");
        TSuccessAuth::FillTokenInfo(comment, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(), "");

        proto = {};
        comment = NExtend::TComment::Create("tokid=token_id;clid=client_id;");
        TSuccessAuth::FillTokenInfo(comment, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "token_info {\n  client_id: \"client_id\"\n  token_id: \"token_id\"\n}\n");

        proto = {};
        comment = NExtend::TComment::Create("tokid=token_id;clid=client_id;devid=device_id;devnm=device_name;scopes=scopes#2;AP=1;");
        TSuccessAuth::FillTokenInfo(comment, proto);
        UNIT_ASSERT_VALUES_EQUAL(proto.DebugString(),
                                 "token_info {\n  client_id: \"client_id\"\n  token_id: \"token_id\"\n  device_id: \"device_id\"\n  device_name: \"device_name\"\n  AP: true\n}\n");
    }

    Y_UNIT_TEST(buildQuery_optional) {
        NExtend::TAuthExtendedEntry entry;

        UNIT_ASSERT(!TSuccessAuth::BuildQuery(entry));

        entry.IpData = NExtend::TIpData{
            .Packed = "123",
            .Unpacked = "456",
            .IsReal = true,
        };
        UNIT_ASSERT(!TSuccessAuth::BuildQuery(entry));

        entry.Row.AuthType = "web";
        entry.Row.Status = "ses_create";
        UNIT_ASSERT(TSuccessAuth::BuildQuery(entry));

        entry.IpData = {};
        UNIT_ASSERT(!TSuccessAuth::BuildQuery(entry));
    }

    Y_UNIT_TEST(buildQuery) {
        NExtend::TAuthExtendedEntry entry;
        entry.Row.AuthType = "web";
        entry.Row.Status = "ses_create";
        entry.IpData = NExtend::TIpData{
            .Packed = "123",
            .Unpacked = "456",
            .IsReal = true,
        };
        entry.UaData = NExtend::TUserAgentData{
            .BrowserName = "kek",
            .BrowserVersion = "ver1",
            .OsName = "lubuntu",
            .OsFamily = "lol",
            .OsVersion = "1.0.1e",
        };
        entry.Comment = NExtend::TComment::Create("tokid=token_id;clid=client_id;devid=device_id;devnm=device_name;scopes=scopes#2;AP=1;");

        auto decode = [](TStringBuf b) {
            return NUtils::Base64ToBin(b);
        };

        UNIT_ASSERT_VALUES_EQUAL(
            *TSuccessAuth::BuildQuery(entry),
            NHbase::TQuery({
                .Row = decode("MDAwAAAAAAAAAAD/////LndlYg=="),
                .Params = {
                    {
                        decode("aToKA3dlYhIFCgMxMjMaCwoDa2VrEgR2ZXIxIhEKB2x1YnVudHUSBjEuMC4xZSovCgljbGllbnRfaWQSCHRva2VuX2lkGglkZXZpY2VfaWQiC2RldmljZV9uYW1lMAE="),
                        decode("MTowLjA="),
                    },
                },
            }));

        entry.Row.AuthType = "oauthcheck";
        UNIT_ASSERT_VALUES_EQUAL(
            *TSuccessAuth::BuildQuery(entry),
            NHbase::TQuery({
                .Row = decode("MDAwAAAAAAAAAAD/////Lm9hdXRoY2hlY2s="),
                .Params = {
                    {
                        decode("aToKCm9hdXRoY2hlY2sSBQoDMTIzKi8KCWNsaWVudF9pZBIIdG9rZW5faWQaCWRldmljZV9pZCILZGV2aWNlX25hbWUwAQ=="),
                        decode("MTowLjA="),
                    },
                },
            }));
    }
}
