#include <passport/infra/daemons/lbchdb/src/crypto/key_ring.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NLbchdb::NCrypto;

Y_UNIT_TEST_SUITE(KeyRing) {
    Y_UNIT_TEST(keyRing) {
        TSimpleKeyRing k(TSimpleKeyRing::TKeys{
            {18, "kek"},
            {129, "illusion"},
        });

        UNIT_ASSERT_VALUES_EQUAL(k.GetOptionalKey(18), "kek");
        UNIT_ASSERT_VALUES_EQUAL(k.GetOptionalKey(129), "illusion");
        UNIT_ASSERT_VALUES_EQUAL(k.GetOptionalKey(100500), IKeyRing::EMPTY_KEY);

        UNIT_ASSERT(k.HasKey(18));
        UNIT_ASSERT(!k.HasKey(100500));

        UNIT_ASSERT_VALUES_EQUAL(k.GetKey(18), "kek");
        UNIT_ASSERT_EXCEPTION_CONTAINS(k.GetKey(100500), yexception, "No key for id '100500'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(k.AddKey(18, "duplicate"), yexception, "Key for id '18' already exists");
        UNIT_ASSERT_VALUES_EQUAL(k.GetKey(18), "kek");

        UNIT_ASSERT_EXCEPTION_CONTAINS(k.AddKey(18, "duplicate", TSimpleKeyRing::EAddKeyPolicy::Remain), yexception, "Key for id '18' changed");
        UNIT_ASSERT_VALUES_EQUAL(k.GetKey(18), "kek");

        UNIT_ASSERT_NO_EXCEPTION(k.AddKey(18, "kek", TSimpleKeyRing::EAddKeyPolicy::Remain));
        UNIT_ASSERT_VALUES_EQUAL(k.GetKey(18), "kek");

        UNIT_ASSERT_NO_EXCEPTION(k.AddKey(100500, "new key"));
        UNIT_ASSERT(k.HasKey(100500));
        UNIT_ASSERT_VALUES_EQUAL(k.GetKey(100500), "new key");
    }

    Y_UNIT_TEST(keyRingFromDir) {
        const TString path = "./some.keys";

        NFs::RemoveRecursive(path);
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSimpleKeyRing::ReadFromFile(path), yexception, R"(can't open "./some.keys")");

        struct TBadCase {
            TString FileContent;
            std::optional<ui64> DefaultKeyId = {};
            TString Error;
        };

        std::vector<TBadCase> badCases({
            TBadCase{
                .FileContent = "salhdfahs",
                .Error = "Failed to parse json object",
            },
            TBadCase{
                .FileContent = R"({"invalid": "d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU="})",
                .Error = "Invalid key id: invalid",
            },
            TBadCase{
                .FileContent = R"({"18": "U638Y2WQ4x14jVWBC+gS1gc="})",
                .Error = "Invalid key size: 17",
            },
            TBadCase{
                .FileContent = R"({
    "18": "d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU=",
    "18": "eik8oMR/sfgUE0O4NEaY/yGgtkfbN3sBCRiyqS0ZDV0="
})",
                .Error = "Duplicated key for id '18'",
            },
        });

        for (const auto& badCase : badCases) {
            TFileOutput(path).Write(badCase.FileContent);
            UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                TSimpleKeyRing(path),
                yexception,
                badCase.Error,
                badCase.FileContent);
        }

        TFileOutput(path).Write(
            R"({
    "18": "d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU=",
    "129": "eik8oMR/sfgUE0O4NEaY/yGgtkfbN3sBCRiyqS0ZDV0="
})");

        TSimpleKeyRing k(path);
        UNIT_ASSERT(k.HasKey(18));
        UNIT_ASSERT_VALUES_EQUAL(NUtils::BinToBase64(k.GetKey(18)), "d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU=");
        UNIT_ASSERT(k.HasKey(129));
        UNIT_ASSERT_VALUES_EQUAL(NUtils::BinToBase64(k.GetKey(129)), "eik8oMR/sfgUE0O4NEaY/yGgtkfbN3sBCRiyqS0ZDV0=");
        UNIT_ASSERT(!k.HasKey(100500));
    }

    Y_UNIT_TEST(keyRingWithEpochFromDir) {
        const TString dir = "./tmp";
        NFs::RemoveRecursive(dir);
        NFs::MakeDirectory(dir);

        TKeyRingWithEpoch k(dir);
        UNIT_ASSERT_EXCEPTION_CONTAINS(k.GetKey(18),
                                       yexception,
                                       R"(can't open "./tmp/18.key")");

        TFileOutput(dir + "/18.key");
        UNIT_ASSERT_EXCEPTION_CONTAINS(k.GetKey(18),
                                       yexception,
                                       R"(FileLoader: file is empty: ./tmp/18.key)");

        {
            TFileOutput f(dir + "/18.key");
            f << "kek";
        }
        UNIT_ASSERT_VALUES_EQUAL("kek", k.GetKey(18));
        UNIT_ASSERT_VALUES_EQUAL("kek", k.GetKey(18));

        NFs::Remove(dir + "/18.key");
        UNIT_ASSERT_VALUES_EQUAL("kek", k.GetKey(18));
    }

    Y_UNIT_TEST(keyRingHolderWithDefaultKey) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TKeyRingHolderWithDefaultKey(nullptr, 10),
            yexception,
            "No keys");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TKeyRingHolderWithDefaultKey(std::make_unique<TSimpleKeyRing>(TSimpleKeyRing::TKeys({{129, "illusion"}})), 18),
            yexception,
            "No key for default key id '18'");

        TKeyRingHolderWithDefaultKey holder(
            std::make_unique<TSimpleKeyRing>(TSimpleKeyRing::TKeys({
                {18, "kek"},
                {129, "illusion"},
            })), 18);

        UNIT_ASSERT_VALUES_EQUAL(18, holder.DefaultKeyID());
        UNIT_ASSERT_VALUES_EQUAL("kek", holder.DefaultKey());
    }
}
