#include <passport/infra/daemons/lbchdb/ut/common.h>

#include <passport/infra/daemons/lbchdb/src/extenders/auth_extender.h>

#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NExtend;

Y_UNIT_TEST_SUITE(ExtendBbAuth) {
    Y_UNIT_TEST(isReal) {
        UNIT_ASSERT(IsRealUserIp(NUtils::TIpAddr()));
        UNIT_ASSERT(IsRealUserIp(NUtils::TIpAddr("8.8.8.8")));

        UNIT_ASSERT(!IsRealUserIp(NUtils::TIpAddr("127.0.0.1"))); // loopback
        UNIT_ASSERT(!IsRealUserIp(NUtils::TIpAddr("10.0.0.1")));  // private
        UNIT_ASSERT(!IsRealUserIp(NUtils::TIpAddr("224.0.0.1"))); // multicast
    }

    Y_UNIT_TEST(isYandex) {
        UNIT_ASSERT(IsYandexIp(IIpReg::TIpInfo{.IsYandex = true, .IsUser = false}));

        UNIT_ASSERT(!IsYandexIp(IIpReg::TIpInfo{.IsYandex = false, .IsUser = false}));
        UNIT_ASSERT(!IsYandexIp(IIpReg::TIpInfo{.IsYandex = false, .IsUser = true}));
        UNIT_ASSERT(!IsYandexIp(IIpReg::TIpInfo{.IsYandex = true, .IsUser = true}));
    }

    Y_UNIT_TEST(IpData_chooseUserIp) {
        struct TDummy {
            TString ProxyIp = "123";
            TString UserIp = "456";
        } row;

        UNIT_ASSERT_VALUES_EQUAL(TIpData::ChooseUserIp(row), "123");

        row.ProxyIp.clear();
        UNIT_ASSERT_VALUES_EQUAL(TIpData::ChooseUserIp(row), "456");

        row.UserIp.clear();
        UNIT_ASSERT_VALUES_EQUAL(TIpData::ChooseUserIp(row), "");

        row.ProxyIp = "789";
        UNIT_ASSERT_VALUES_EQUAL(TIpData::ChooseUserIp(row), "789");
    }

    Y_UNIT_TEST(IpData_create) {
        TTestGeobase geobase;
        geobase.As = "some as";
        geobase.RegionId = 42;
        TTestIpReg ipreg;
        ipreg.Info = IIpReg::TIpInfo{.IsYandex = true, .IsUser = false};

        // bad ip
        UNIT_ASSERT(!TIpData::Create(geobase, ipreg, ""));
        UNIT_ASSERT(!TIpData::Create(geobase, ipreg, "asdfg"));

        UNIT_ASSERT_VALUES_EQUAL(
            *TIpData::Create(geobase, ipreg, "127.0.0.1"),
            TIpData({
                .Packed = TString(std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x7f, 0x00, 0x00, 0x01})),
                .PackedShortest = TString(std::string({0x7f, 0x00, 0x00, 0x01})),
                .Unpacked = "127.0.0.1",
                .IsReal = false,
                .IsYandex = true,
            }));

        UNIT_ASSERT_VALUES_EQUAL(
            *TIpData::Create(geobase, ipreg, "0.0.0.0"),
            TIpData({
                .Packed = TString(std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x00, 0x00, 0x00, 0x00})),
                .PackedShortest = TString(std::string({0x00, 0x00, 0x00, 0x00})),
                .Unpacked = "0.0.0.0",
                .IsReal = true,
                .IsYandex = true,
            }));

        UNIT_ASSERT_VALUES_EQUAL(
            *TIpData::Create(geobase, ipreg, "8.8.8.8"),
            TIpData({
                .GeoId = 42,
                .As = "some as",
                .Packed = TString(std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x08, 0x08, 0x08, 0x08})),
                .PackedShortest = TString(std::string({0x08, 0x08, 0x08, 0x08})),
                .Unpacked = "8.8.8.8",
                .IsReal = true,
                .IsYandex = true,
            }));

        geobase.RegionId = {};
        geobase.As = {};
        ipreg.Info = {};

        UNIT_ASSERT_VALUES_EQUAL(
            *TIpData::Create(geobase, ipreg, "8.8.8.8"),
            TIpData({
                .Packed = TString(std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x08, 0x08, 0x08, 0x08})),
                .PackedShortest = TString(std::string({0x08, 0x08, 0x08, 0x08})),
                .Unpacked = "8.8.8.8",
                .IsReal = true,
                .IsYandex = false,
            }));
    }

    Y_UNIT_TEST(UserAgentData) {
        TTestUaTraits uatraits;

        // no data
        UNIT_ASSERT(!TUserAgentData::Create(uatraits, ""));
        UNIT_ASSERT(!TUserAgentData::Create(uatraits, "some browser"));
        uatraits.Data = {{}};
        UNIT_ASSERT(!TUserAgentData::Create(uatraits, "some browser"));

        uatraits.Data = {IUaTraits::TData::TMap{
            {"key", "value"},
        }};
        UNIT_ASSERT_VALUES_EQUAL(
            *TUserAgentData::Create(uatraits, "some browser"),
            TUserAgentData({}));

        uatraits.Data = {IUaTraits::TData::TMap{
            {"BrowserName", "kek"},
            {"OSFamily", "lol"},
        }};
        UNIT_ASSERT_VALUES_EQUAL(
            *TUserAgentData::Create(uatraits, "some browser"),
            TUserAgentData({
                .BrowserName = "kek",
                .OsFamily = "lol",
            }));

        uatraits.Data = {IUaTraits::TData::TMap{
            {"BrowserName", "kek"},
            {"BrowserVersion", "ver1"},
            {"OSName", "lubuntu"},
            {"OSFamily", "lol"},
            {"OSVersion", "1.0.1e"},
        }};
        UNIT_ASSERT_VALUES_EQUAL(
            *TUserAgentData::Create(uatraits, "some browser"),
            TUserAgentData({
                .BrowserName = "kek",
                .BrowserVersion = "ver1",
                .OsName = "lubuntu",
                .OsFamily = "lol",
                .OsVersion = "1.0.1e",
            }));
    }

    Y_UNIT_TEST(Laas) {
        std::optional<NExtend::TLaasInfo> laas = NExtend::TLaasInfo::Create("60.166892=24.943592=15000=region");
        NExtend::TLaasInfo info{
            .Latitude = 60.166892,
            .Longitude = 24.943592,
            .Accuracy = 15000,
            .Precision = "region",
        };
        UNIT_ASSERT_VALUES_EQUAL(laas, info);

        UNIT_ASSERT(!NExtend::TLaasInfo::Create("60.16689224.943592=15000=region"));
        UNIT_ASSERT(!NExtend::TLaasInfo::Create("60.16a892=24.943592=15000=region"));
        UNIT_ASSERT(!NExtend::TLaasInfo::Create("60.166892=24.9!3592=15000=region"));
        UNIT_ASSERT(!NExtend::TLaasInfo::Create("60.166892=24.943592=15000.1=region"));
        UNIT_ASSERT(!NExtend::TLaasInfo::Create("60.166892=24.943592=15000=region=asdf"));
    }

    Y_UNIT_TEST(Comment) {
        UNIT_ASSERT_VALUES_EQUAL(TComment::Create("key=value;").PrintDebug(), "");
        UNIT_ASSERT_VALUES_EQUAL(TComment::Create("tokid;").PrintDebug(), "");
        UNIT_ASSERT_VALUES_EQUAL(TComment::Create("tokid_value;").PrintDebug(), "");

        UNIT_ASSERT_VALUES_EQUAL(TComment::Create("tokid=value1=asdad;").PrintDebug(),
                                 "'tokid'->'value1=asdad'; ");

        const TComment comment = TComment::Create("tokid=value1;clid=value2;devid=value3;devnm=value4;scope=value5;AP=value6;asrc=value7;pos=value8;");
        UNIT_ASSERT_VALUES_EQUAL(comment.PrintDebug(),
                                 "'tokid'->'value1'; 'clid'->'value2'; 'devid'->'value3'; 'devnm'->'value4'; 'scope'->'value5'; 'AP'->'value6'; 'asrc'->'value7'; 'pos'->'value8'; ");
        UNIT_ASSERT_VALUES_EQUAL(comment.Get(TComment::EKey::TokenId), "value1");
        UNIT_ASSERT_VALUES_EQUAL(comment.Get(TComment::EKey::ClientId), "value2");
    }

    Y_UNIT_TEST(BbAuthExtendedRow) {
        TTestGeobase geobase;
        geobase.As = "some as";
        geobase.RegionId = 42;

        TTestIpReg ipreg;
        ipreg.Info = IIpReg::TIpInfo{.IsYandex = true, .IsUser = false};

        TTestUaTraits uatraits;
        uatraits.Data = {IUaTraits::TData::TMap{
            {"BrowserName", "kek"},
            {"BrowserVersion", "ver1"},
            {"OSName", "lubuntu"},
            {"OSFamily", "lol"},
            {"OSVersion", "1.0.1e"},
        }};

        TAuthExtender ext(geobase, ipreg, uatraits);

        NParser::TAuthRow r;
        r.ProxyIp = "8.8.8.8";
        r.UserIp = "127.0.0.0";
        r.Comment = "tokid=value1;clid=value2;devid=value3;devnm=value4;scope=value5;pos=60.166892=24.943592=15000=region;ttl=5";
        r.UserAgent = "some agent";

        TAuthExtendedEntry entry = ext.Process(NParser::TAuthRow(r));
        UNIT_ASSERT_VALUES_EQUAL(r, entry.Row);
        UNIT_ASSERT(entry.IpData);
        UNIT_ASSERT_VALUES_EQUAL(
            *entry.IpData,
            TIpData({
                .GeoId = 42,
                .As = "some as",
                .Packed = TString(std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x08, 0x08, 0x08, 0x08})),
                .PackedShortest = TString(std::string({0x08, 0x08, 0x08, 0x08})),
                .Unpacked = "8.8.8.8",
                .IsReal = true,
                .IsYandex = true,
            }));
        UNIT_ASSERT(entry.UaData);
        UNIT_ASSERT_VALUES_EQUAL(
            *entry.UaData,
            TUserAgentData({
                .BrowserName = "kek",
                .BrowserVersion = "ver1",
                .OsName = "lubuntu",
                .OsFamily = "lol",
                .OsVersion = "1.0.1e",
            }));
        UNIT_ASSERT(entry.Laas);
        UNIT_ASSERT_VALUES_EQUAL(*entry.Laas, NExtend::TLaasInfo({
                                                  .Latitude = 60.166892,
                                                  .Longitude = 24.943592,
                                                  .Accuracy = 15000,
                                                  .Precision = "region",
                                              }));
        UNIT_ASSERT_VALUES_EQUAL(
            entry.Comment,
            TComment::Create("tokid=value1;clid=value2;devid=value3;devnm=value4;scope=value5;pos=60.166892=24.943592=15000=region;ttl=5"));
    }
}
