#include <passport/infra/daemons/lbchdb/ut/common.h>

#include <passport/infra/daemons/lbchdb/src/extenders/event_extender.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NParser;
using namespace NPassport::NLbchdb::NExtend;
using namespace NPassport::NLb;
using namespace std::string_literals;

TString PackedIpToHex(TStringBuf s) {
    TStringBuilder o;
    for (const char c : s) {
        o << IntToString<16>(ui8(c));
    }
    return o;
}

Y_UNIT_TEST_SUITE(EventExtender) {
    class TEventExtenderHolder {
    public:
        TEventExtenderHolder()
            : Extender(Geobase, IpReg)
        {
        }

        void SetIpData(const std::optional<TIpData>& data) {
            if (!data) {
                Geobase.As.reset();
                Geobase.RegionId.reset();
                IpReg.Info.IsYandex = false;
                IpReg.Info.IsUser = false;
                return;
            }

            Geobase.As = data->As;
            Geobase.RegionId = data->GeoId;

            IpReg.Info.IsYandex = data->IsYandex;
            IpReg.Info.IsUser = data->IsReal;
        }

        TTestGeobase Geobase;
        TTestIpReg IpReg;

        TEventExtender Extender;
    };

    Y_UNIT_TEST(IpParse) {
        TEventExtenderHolder e;

        struct TTestCase {
            TEventRow Row;
            TEventExtendedEntry Entry;
        };

        std::list<TTestCase> tests;
        TTestCase testCase{
            .Row = {},
            .Entry = {},
        };
        tests.push_back(std::move(testCase));

        testCase = {
            .Row = {
                .EventValue = "some data",
                .UserIp = "::ffff:asdf",
                .ProxyIp = "::ffff:fdas",

            },
            .Entry = {.Row = {
                          .EventValue = "some data",
                          .UserIp = "asdf",
                          .ProxyIp = "fdas",
                      }},
        };
        tests.push_back(std::move(testCase));

        testCase = {
            .Row = {
                .ProxyIp = "234.234.234.234",
            },
            .Entry = {
                .Row = {.ProxyIp = "234.234.234.234"},
                .IpData = TIpData{
                    .GeoId = 60,
                    .As = "asdf1",
                    .Packed = TString("\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xea\xea\xea\xea"s),
                    .PackedShortest = TString("\xea\xea\xea\xea"s),
                    .Unpacked = "234.234.234.234",
                    .IsReal = false,
                    .IsYandex = true,
                },
            },
        };
        tests.push_back(std::move(testCase));

        testCase = {
            .Row = {
                .UserIp = "some ip",
                .ProxyIp = "::ffff:123.231.132.213",
            },
            .Entry = {
                .Row = {.UserIp = "some ip", .ProxyIp = "123.231.132.213"},
                .IpData = TIpData{
                    .GeoId = 61,
                    .As = "asdf2",
                    .Packed = TString("\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\x7b\xe7\x84\xd5"s),
                    .PackedShortest = TString("\x7b\xe7\x84\xd5"s),
                    .Unpacked = "::ffff:123.231.132.213",
                    .IsReal = true,
                    .IsYandex = false,
                },
            },
        };
        tests.push_back(std::move(testCase));

        testCase = {
            .Row = {
                .UserIp = "2001:0db8:85a3:0000:0000:8a2e:0370:7334",
            },
            .Entry = {
                .Row = {.UserIp = "2001:0db8:85a3:0000:0000:8a2e:0370:7334"},
                .IpData = TIpData{
                    .GeoId = 62,
                    .As = "asdf3",
                    .Packed = NUtils::TIpAddr("2001:0db8:85a3:0000:0000:8a2e:0370:7334").ToPackedStringV6(),
                    .PackedShortest = NUtils::TIpAddr("2001:0db8:85a3:0000:0000:8a2e:0370:7334").ToPackedStringShortest(),
                    .Unpacked = "2001:0db8:85a3:0000:0000:8a2e:0370:7334",
                    .IsReal = true,
                    .IsYandex = false,
                },
            },
        };
        tests.push_back(std::move(testCase));

        for (auto& test : tests) {
            e.SetIpData(test.Entry.IpData);
            UNIT_ASSERT_VALUES_EQUAL(e.Extender.Process(std::move(test.Row)), test.Entry);
        }
    }
}
