#include <passport/infra/daemons/lbchdb/src/extenders/sender_delivery_extender.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NExtend;

Y_UNIT_TEST_SUITE(SenderDeliveryExtender) {
    static const TString EMPTY_RESP = R"(
{
  "count": "0",
  "total_count": "0",
  "uids": []
})";
    static const TString USERINFO_WITH_USERS = R"(
{
  "users": [{
      "id": "17145248",
      "uid": {
        "value": "17145248",
        "lite": false,
        "hosted": false
      },
      "login": "asdzxcq",
      "have_password": true,
      "have_hint": true,
      "karma": {"value": 0},
      "karma_status": {"value": 0}
  }]
})";
    static const TString EMAIL_BINDINGS_WITH_USERS = R"(
{
  "count": "1",
  "total_count": "1",
  "uids": [
    "71001",
    "71002",
    "71003"
  ]
})";

    class TTestClient: public IBlackboxClient {
    public:
        TRequestHandle SendRequest(const TRequest& req) override {
            ++Sends;
            if (ThrowCantGetCon) {
                ThrowCantGetCon = false;
                throw NDbPool::TCantGetConnection();
            }
            if (ThrowSmth) {
                ThrowSmth = false;
                throw yexception();
            }
            return {.Req = req};
        }

        TString WaitResponse(TRequestHandle&) const override {
            ++Waits;
            if (ThrowOnWait) {
                throw yexception();
            }
            return Resp;
        }

        bool ThrowCantGetCon = false;
        bool ThrowSmth = false;
        bool ThrowOnWait = false;
        size_t Sends = 0;
        mutable size_t Waits = 0;
        TString Resp = EMPTY_RESP;
    };

    class TTestExtender: public TSenderDeliveryExtender {
    public:
        explicit TTestExtender(IBlackboxClient& bb,
                               const TSenderDeliveryExtenderSettings& settings = {})
            : TSenderDeliveryExtender(bb, BbErrors_, settings)
        {
        }

        using TSenderDeliveryExtender::EmailBindings_;
        using TSenderDeliveryExtender::Userinfo_;

    private:
        NUnistat::TSignalDiff<> BbErrors_ = {"blackbox.errors"};
    };

    Y_UNIT_TEST(addResult) {
        TTestClient bb;

        TTestExtender ext(bb);
        ext.Reserve(17);

        UNIT_ASSERT_VALUES_EQUAL(
            TSenderDeliveryExtender::TResult({}),
            ext.Finish());

        ext.AddResult(100, 200, 300);
        ext.AddResult(101, 201, 301);
        ext.AddResult(102, 202, 302);

        UNIT_ASSERT_VALUES_EQUAL(
            TSenderDeliveryExtender::TResult({
                TSenderDeliveryExtendedEntry{
                    .Uid = 100,
                    .UnsubscribeList = 200,
                    .Ts = 300,
                },
                TSenderDeliveryExtendedEntry{
                    .Uid = 101,
                    .UnsubscribeList = 201,
                    .Ts = 301,
                },
                TSenderDeliveryExtendedEntry{
                    .Uid = 102,
                    .UnsubscribeList = 202,
                    .Ts = 302,
                },
            }),
            ext.Finish());
    }

    Y_UNIT_TEST(tryWait) {
        TTestClient bb;

        TTestExtender ext(bb);

        bool throwParseExc = false;
        size_t procs = 0;
        auto proc = [&](const TString&, TSenderDeliveryExtender::TRequest&) {
            ++procs;
            if (throwParseExc) {
                throw TBlackboxClient::TParseException();
            }
        };

        std::deque<TSenderDeliveryExtender::TRequest> reqs;
        std::optional<TSenderDeliveryExtender::TRequest> toResend;

        UNIT_ASSERT_NO_EXCEPTION(toResend = ext.TryWait(reqs, proc));
        UNIT_ASSERT(!toResend);
        UNIT_ASSERT_VALUES_EQUAL(0, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL(0, procs);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);

        reqs.push_back({.BbReq = {.Req = {.Body = "req_body#1"}}});
        reqs.push_back({.BbReq = {.Req = {.Body = "req_body#2"}}});
        reqs.push_back({.BbReq = {.Req = {.Body = "req_body#3"}}});
        reqs.push_back({.BbReq = {.Req = {.Body = "req_body#4"}}});

        UNIT_ASSERT_NO_EXCEPTION(toResend = ext.TryWait(reqs, proc));
        UNIT_ASSERT(!toResend);
        UNIT_ASSERT_VALUES_EQUAL(3, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#2", reqs.front().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, procs);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Waits);
        procs = 0;
        bb.Waits = 0;

        bb.ThrowOnWait = true;
        UNIT_ASSERT_NO_EXCEPTION(toResend = ext.TryWait(reqs, proc));
        UNIT_ASSERT(toResend);
        UNIT_ASSERT_VALUES_EQUAL("req_body#2", toResend->BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(2, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#3", reqs.front().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(0, procs);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Waits);
        bb.Waits = 0;

        bb.ThrowOnWait = false;
        throwParseExc = true;
        UNIT_ASSERT_NO_EXCEPTION(toResend = ext.TryWait(reqs, proc));
        UNIT_ASSERT(toResend);
        UNIT_ASSERT_VALUES_EQUAL("req_body#3", toResend->BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#4", reqs.front().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, procs);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Waits);
        procs = 0;
        bb.Waits = 0;
    }

    Y_UNIT_TEST(tryAdd) {
        TTestClient bb;

        TTestExtender ext(bb);
        for (int idx = 0; idx < 10; ++idx) {
            ext.Userinfo_.push_back({.BbReq{.Req = {.Body = NUtils::CreateStr("req_body_userinfo#", idx)}}});
            ext.EmailBindings_.push_back({.BbReq{.Req = {.Body = NUtils::CreateStr("req_body_ebinds#", idx)}}});
        }

        std::deque<TSenderDeliveryExtender::TRequest> reqs;

        bool status = false;
        UNIT_ASSERT_NO_EXCEPTION(status = ext.TryAdd(100, 200, reqs, {.Body = "req_body#1"}));
        UNIT_ASSERT(status);
        UNIT_ASSERT_VALUES_EQUAL(1, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#1", reqs.back().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);
        bb.Sends = 0;

        bb.ThrowCantGetCon = true;
        UNIT_ASSERT_NO_EXCEPTION(status = ext.TryAdd(100, 200, reqs, {.Body = "req_body#2"}));
        UNIT_ASSERT(!status);
        UNIT_ASSERT_VALUES_EQUAL(1, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#1", reqs.back().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Waits);
        bb.Sends = 0;
        bb.Waits = 0;

        bb.ThrowCantGetCon = true;
        UNIT_ASSERT_NO_EXCEPTION(
            status = ext.TryAdd(100,
                                200,
                                reqs,
                                {.Body = "req_body#2"},
                                TSenderDeliveryExtender::EWaitResponses::False));
        UNIT_ASSERT(!status);
        UNIT_ASSERT_VALUES_EQUAL(1, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#1", reqs.back().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);
        bb.Sends = 0;

        bb.ThrowSmth = true;
        UNIT_ASSERT_NO_EXCEPTION(status = ext.TryAdd(100, 200, reqs, {.Body = "req_body#2"}));
        UNIT_ASSERT(!status);
        UNIT_ASSERT_VALUES_EQUAL(1, reqs.size());
        UNIT_ASSERT_VALUES_EQUAL("req_body#1", reqs.back().BbReq.Req.Body);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);
        bb.Sends = 0;
    }

    Y_UNIT_TEST(waitSomeResponses) {
        TTestClient bb;

        TTestExtender ext(bb);

        UNIT_ASSERT_NO_EXCEPTION(ext.WaitSomeResponses());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);

        for (int idx = 0; idx < 10; ++idx) {
            ext.Userinfo_.push_back({.BbReq{.Req = {.Body = NUtils::CreateStr("req_body_userinfo#", idx)}}});
            ext.EmailBindings_.push_back({.BbReq{.Req = {.Body = NUtils::CreateStr("req_body_ebinds#", idx)}}});
        }

        UNIT_ASSERT_NO_EXCEPTION(ext.WaitSomeResponses());
        UNIT_ASSERT_VALUES_EQUAL(9, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(9, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Waits);
        bb.Waits = 0;

        bb.ThrowOnWait = true;
        UNIT_ASSERT_NO_EXCEPTION(ext.WaitSomeResponses());
        UNIT_ASSERT_VALUES_EQUAL(9, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(9, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Waits);
        bb.Sends = 0;
        bb.Waits = 0;

        ext.EmailBindings_.clear();
        bb.ThrowOnWait = true;
        UNIT_ASSERT_NO_EXCEPTION(ext.WaitSomeResponses());
        UNIT_ASSERT_VALUES_EQUAL(9, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Waits);
        bb.Sends = 0;
        bb.Waits = 0;

        bb.ThrowOnWait = false;
        bb.Resp = USERINFO_WITH_USERS;
        UNIT_ASSERT_VALUES_EQUAL(9, ext.Finish().size());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(9, bb.Waits);
        bb.Waits = 0;

        ext.EmailBindings_.push_back({.BbReq{.Req = {.Body = "req_body_ebinds#N"}}});
        bb.Resp = EMAIL_BINDINGS_WITH_USERS;
        UNIT_ASSERT_VALUES_EQUAL(3, ext.Finish().size());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(1, bb.Waits);
    }

    Y_UNIT_TEST(add) {
        TTestClient bb;

        TTestExtender ext(bb, TSenderDeliveryExtenderSettings{
                                  .InflightReqsLimit = 4,
                                  .SleepOnFailMin = TDuration::MilliSeconds(1),
                                  .SleepOnFailMax = TDuration::MilliSeconds(5),
                              });

        ext.Add(NParser::TSenderDeliveryRow{.RecipientEmail = "100", .Ts = 300});
        UNIT_ASSERT_VALUES_EQUAL(1, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(1, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);
        bb.Sends = 0;

        ext.Add(NParser::TSenderDeliveryRow{.RecipientEmail = "101", .Ts = 301});
        UNIT_ASSERT_VALUES_EQUAL(2, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Waits);
        bb.Sends = 0;

        ext.Add(NParser::TSenderDeliveryRow{.RecipientEmail = "102", .Ts = 302});
        UNIT_ASSERT_VALUES_EQUAL(2, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(2, bb.Waits);
        bb.Sends = 0;
        bb.Waits = 0;

        UNIT_ASSERT_VALUES_EQUAL(0, ext.Finish().size());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.Userinfo_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, ext.EmailBindings_.size());
        UNIT_ASSERT_VALUES_EQUAL(0, bb.Sends);
        UNIT_ASSERT_VALUES_EQUAL(4, bb.Waits);
    }
}
