#include <passport/infra/daemons/lbchdb/ut/common.h>

#include <passport/infra/daemons/lbchdb/src/parsers/auth_parser.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NParser;

Y_UNIT_TEST_SUITE(AuthParser) {
    class TTestRow: public NRowParsers::TCsvRow { // NOLINT(bugprone-exception-escape)
    public:
        using TCsvRow::TCsvRow;

        template <typename T>
        TTestRow Clone(T idx, TStringBuf val) const {
            TTestRow res(*this);
            res.Fields_.at((size_t)idx) = NRowParsers::TField(val);
            return res;
        }
    };

    Y_UNIT_TEST(TAuthRow) {
        SetTimeZone();
        TTestRow goodCtx("1 2020-04-13T14:29:01.997207+03 7F bb 1130000002760050 - - oauthcheck successful clid=7a663d0b83664d9f9bc70d465fb9868b;tokid=14549038;devid=6781d2c10844a8074743ded4d0be6e9d;devnm=Emulator;scope=oauth:grant_xtoken; 2a02:6b8:c01:b80:0:663:5838:75a9 - - - - -");

        TAuthRow row;

        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse({"1 2020-04-13T14:29:01.997207+03 7F bb"}),
                                       yexception,
                                       "unexpected fields count: 4");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse(goodCtx.Clone(EAuthRow::Version, "2")),
                                       yexception,
                                       "unexpected version");

        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse(goodCtx.Clone(EAuthRow::Time, "2020-04-13Tasdadss9:01.997207+03")),
                                       yexception,
                                       "invalid time");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse(goodCtx.Clone(EAuthRow::Uid, "113as0002760050")),
                                       yexception,
                                       "invalid uid");

        UNIT_ASSERT(row.TryParse(NRowParsers::TCsvRow(goodCtx)));
        UNIT_ASSERT_VALUES_EQUAL(row.Timestamp, TInstant::ParseIso8601("2020-04-13T11:29:01.997207Z"));
        UNIT_ASSERT_VALUES_EQUAL(row.HostId, 127);
        UNIT_ASSERT_VALUES_EQUAL(row.Uid, 1130000002760050ul);
        UNIT_ASSERT_VALUES_EQUAL(row.Login, "");
        UNIT_ASSERT_VALUES_EQUAL(row.Sid, "");
        UNIT_ASSERT_VALUES_EQUAL(row.AuthType, "oauthcheck");
        UNIT_ASSERT_VALUES_EQUAL(row.Status, "successful");
        UNIT_ASSERT_VALUES_EQUAL(row.Comment, "clid=7a663d0b83664d9f9bc70d465fb9868b;tokid=14549038;devid=6781d2c10844a8074743ded4d0be6e9d;devnm=Emulator;scope=oauth:grant_xtoken;");
        UNIT_ASSERT_VALUES_EQUAL(row.UserIp, "2a02:6b8:c01:b80:0:663:5838:75a9");
        UNIT_ASSERT_VALUES_EQUAL(row.ProxyIp, "");
        UNIT_ASSERT_VALUES_EQUAL(row.YandexUid, "");
        UNIT_ASSERT_VALUES_EQUAL(row.UserAgent, "");

        row = TAuthRow();
        UNIT_ASSERT(!row.TryParse(goodCtx.Clone(EAuthRow::Uid, "-")));

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Login, "Vasya")));
        UNIT_ASSERT_VALUES_EQUAL(row.Login, "Vasya");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Sid, "foo")));
        UNIT_ASSERT_VALUES_EQUAL(row.Sid, "");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Type, "imap")));
        UNIT_ASSERT_VALUES_EQUAL(row.AuthType, "imap");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Status, "failed")));
        UNIT_ASSERT_VALUES_EQUAL(row.Status, "failed");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Comment, "some mega comment")));
        UNIT_ASSERT_VALUES_EQUAL(row.Comment, "some mega comment");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::UserIp, "-")));
        UNIT_ASSERT_VALUES_EQUAL(row.UserIp, "");
        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::UserIp, "::ffff:kek")));
        UNIT_ASSERT_VALUES_EQUAL(row.UserIp, "kek");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::ProxyIp, "::ffff:127.0.0.1")));
        UNIT_ASSERT_VALUES_EQUAL(row.ProxyIp, "127.0.0.1");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::Yandexuid, "bar")));
        UNIT_ASSERT_VALUES_EQUAL(row.YandexUid, "bar");

        UNIT_ASSERT(row.TryParse(goodCtx.Clone(EAuthRow::UserAgent, "ua")));
        UNIT_ASSERT_VALUES_EQUAL(row.UserAgent, "ua");
    }

    Y_UNIT_TEST(parse) {
        SetTimeZone();
        TTestRow ctx("1 2021-02-16T19:41:40.536827+03 174 bb 197134077 - - oauthcheck successful clid=f8cab64f154b4c8e96f92dac8becfcaa;tokid=1603903192049197134077;devid=ea641b05c9eb32d41689de9e3d73022d;devnm=Honor+9+Lite;scope=mobile:all,mobmail:all,music:read,yadisk:all,yadisk:browser_sync,yataxi:pay,yataxi:read,yataxi:write,cloud_api.data:user_data,cloud_api.data:app_data,yamb:all,quasar:all,quasar:pay,passport:bind_email,quasar:glagol,payments:all,login:default_phone;tvm_id=2005873; 91.215.232.239 - - - - -");

        TAuthRow row;
        UNIT_ASSERT(row.TryParse(std::move(ctx)));
    }
}
