#include <passport/infra/daemons/lbchdb/src/parsers/event_parser.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NCrypto;
using namespace NPassport::NLbchdb::NParser;
using namespace NPassport::NLb;

Y_UNIT_TEST_SUITE(EventParser) {
    class TEventParserTest: public TEventParser {
    public:
        TEventParserTest(const TDecryptor& decryptor)
            : TEventParser(decryptor)
                  {};

        using TEventParser::TryParse;
    };

    class TEventParserHolder {
    public:
        TEventParserHolder()
            : Decryptor(TDecryptor(std::make_unique<TSimpleKeyRing>(TSimpleKeyRing::TKeys{
                  {666, NUtils::Base64ToBin("S0VZWUtFWVlLRVlZS0VZWUtFWVlLRVlZS0VZWUtFWVk=")},
              })))
            , Parser(Decryptor)
        {
        }
        TDecryptor Decryptor;
        TEventParserTest Parser;
    };

    Y_UNIT_TEST(ParseBadTData) {
        struct TTestCase {
            TStringBuf Line;
            TStringBuf Exception;
        };

        std::list<TTestCase> tests{
            TTestCase{
                .Line = "12021-05-25T15:17:24.002968+03 47 passport 4072886674 action account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Invalid field count in entry",
            },
            TTestCase{
                .Line = "0 2021-05-25T15:17:24.002968+03 47 passport 4072886674 action account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Invalid version",
            },
            TTestCase{
                .Line = "a 2021-05-25T15:17:24.002968+03 47 passport 4072886674 action account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Invalid version",
            },
            TTestCase{
                .Line = "1 2021-05-25T15:60:24.00296803 47 passport 4072886674 action account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Invalid rfctime",
            },
            TTestCase{
                .Line = "1 2021-05-25T15:17:24.002968+03 47 passport -1 action account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Invalid UID",
            },
            TTestCase{
                .Line = "1 2021-05-25T15:17:24.002968+03 47 passport 4072886674 - account_register 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "Event name cannot be empty",
            },
            TTestCase{
                .Line = "2 2021-05-25T15:17:24.002968+03 47 passport 4072886674 info.password DEFINITELY_ENCRYPTED_DATA 37.9.101.12 37.9.101.11 12345 admin some_comment",
                .Exception = "invalid format of encrypted string",
            },
        };

        TEventParserHolder p;

        for (const TTestCase& test : tests) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Parser.TryParse(test.Line), yexception, test.Exception);
        }
    }

    Y_UNIT_TEST(ParseGoodData) {
        struct TTestCase {
            TStringBuf Line;
            TEventRow Row;
        };

        std::list<TTestCase> goodLines;

        TTestCase testCase{
            .Line = "1 2021-05-25T15:17:24.002968+03 47 passport 4072886674 action account_register 37.9.101.12 INVALID_IP 1227346283746283746238345 admin some_comment",
            .Row = TEventRow{
                .ClientName = "passport",
                .EventValue = "account_register",
                .UserIp = "37.9.101.12",
                .ProxyIp = "INVALID_IP",
                .YandexUid = "1227346283746283746238345",
                .AdminLogin = "admin",
                .Comment = "some_comment",
            },
        };

        testCase.Row.Timestamp = TInstant::ParseIso8601("2021-05-25T15:17:24.002968+03");
        testCase.Row.Name = "action";
        testCase.Row.HostId = 71;
        testCase.Row.Uid = 4072886674;
        goodLines.push_back(std::move(testCase));

        testCase = {
            .Line = "1 2021-05-25T15:19:24.002968+03 234 - 4072336674 action:2 - - - - - -",
            .Row = TEventRow{},
        };

        testCase.Row.Timestamp = TInstant::ParseIso8601("2021-05-25T15:19:24.002968+03");
        testCase.Row.Name = "action:2";
        testCase.Row.HostId = 564;
        testCase.Row.Uid = 4072336674;
        goodLines.push_back(std::move(testCase));

        testCase = {
            .Line = "2 2021-05-25T15:29:24.002968+03 ff - 40774 info.password gNMH92uXi1kjCNUc:uT4=:key_number=666:8yLlAyi2Fqk8TzzXd+ZACg== - - - - -",
            .Row = TEventRow{
                .EventValue = "{}",
            },
        };

        testCase.Row.Timestamp = TInstant::ParseIso8601("2021-05-25T15:29:24.002968+03");
        testCase.Row.Name = "info.password";
        testCase.Row.HostId = 255;
        testCase.Row.Uid = 40774;
        goodLines.push_back(std::move(testCase));

        testCase = {
            .Line = "2 2019-05-25T15:29:24.002968+03 acdc - 123456 info.hinta 6ThMEri8ZUsiF4Gc:asIQnfAyrZl/ZA==:key_number=666:sGUNScY9a/vFyKJNlTrCEg== 2a02:6b8:0:123:fc76:c9ff:c666:a5bc ::ffff:127.0.0.1 - - -",
            .Row = TEventRow{
                .EventValue = "{\"a\": 234}",
                .UserIp = "2a02:6b8:0:123:fc76:c9ff:c666:a5bc",
                .ProxyIp = "::ffff:127.0.0.1",
            },
        };

        testCase.Row.Timestamp = TInstant::ParseIso8601("2019-05-25T15:29:24.002968+03");
        testCase.Row.Name = "info.hinta";
        testCase.Row.HostId = 44252;
        testCase.Row.Uid = 123456;
        goodLines.push_back(std::move(testCase));

        TEventParserHolder p;
        for (const auto& test : goodLines) {
            UNIT_ASSERT_VALUES_EQUAL(p.Parser.TryParse(test.Line), test.Row);
        }
    }
}
