#include <passport/infra/daemons/lbchdb/src/parsers/oauth_parser.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NParser;
using namespace NPassport::NLb;

Y_UNIT_TEST_SUITE(OAuthParser) {
    class TOAuthParserTest: public TOAuthParser {
    public:
        using TOAuthParser::TryParse;
    };

    Y_UNIT_TEST(ParseBadTData) {
        struct TTestCase {
            TString Line;
            TString Exception;
        };

        const std::vector<TTestCase> tests{
            {"line", "Incorrect tskv format"},

            {"v=2\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "Unsupported version"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=jhgf\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "Invalid UID"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'uid'"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)",
             "no field 'timestamp'"},

            {"v=1\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'action'"},

            {"v=1\taction=create\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'target'"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "scopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'client_id'"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'token_id'"},

            {"v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
             "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
             "consumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
             "no field 'user_ip'"},
        };

        TOAuthParserTest p;
        for (const auto& test : tests) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.TryParse(test.Line), yexception, test.Exception);
        }
    }

    Y_UNIT_TEST(ParseGoodTData) {
        struct TTestCase {
            TString Line;
            std::optional<TOAuthRow> Row;
        };

        std::vector<TTestCase> tests;

        TTestCase tc = {
            .Line = "v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
                    "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
                    "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
            .Row = TOAuthRow{
                .Timestamp = (TInstant::Seconds(1622707526) + TDuration::MicroSeconds(80980)),
                .TokenId = "26452866",
                .ClientId = "1898303434af426282ce7d45c89629c1",
                .Scopes = "oauth:grant_xtoken",
                .UserIp = "77.79.1.11",
                .Uid = 4073655694,
            },
        };
        tests.push_back(std::move(tc));

        tc = TTestCase{
            .Line = "v=1\taction=create\ttarget=token\ttoken_type=stateless\tgrant_type=passport_assertion\tuid=4073653668\ttoken_id=16227074630074073653668\t"
                    "client_id=52cb86ab8134497d9eab9ebf361345e2\tscopes=yauber:request\thas_alias=0\tpassword_passed=0\t"
                    "user_ip=127.247.212.236\tconsumer_ip=2a02:6b8:c01:710:8000:611:0:11\tuser_agent=python-requests/2.25.1\ttimestamp=1622707463.009011",
            .Row = {},
        };
        tests.push_back(std::move(tc));

        tc = TTestCase{
            .Line = "v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4034913654\ttoken_id=16382761\t"
                    "client_id=f171112adc6a40f59dd46c9822ef1168\tscopes=yataxi:write,payments:all,yamb:all,yadisk:browser_sync,yataxi:pay,yataxi:read\t"
                    "has_alias=0\tpassword_passed=1\tuser_ip=2a02:6b8:0:3304:1809:80b7:7015:2f54\t"
                    "consumer_ip=2a02:6b8:0:3304:1809:80b7:7015:2f54\tuser_agent=curl/7.64.1\ttimestamp=1622699101.410735",
            .Row = TOAuthRow{
                .Timestamp = (TInstant::Seconds(1622699101) + TDuration::MicroSeconds(410735)),
                .TokenId = "16382761",
                .ClientId = "f171112adc6a40f59dd46c9822ef1168",
                .Scopes = "yataxi:write,payments:all,yamb:all,yadisk:browser_sync,yataxi:pay,yataxi:read",
                .UserIp = "2a02:6b8:0:3304:1809:80b7:7015:2f54",
                .Uid = 4034913654,
            },
        };
        tests.push_back(std::move(tc));

        TOAuthParserTest p;

        for (const auto& test : tests) {
            UNIT_ASSERT_VALUES_EQUAL_C(p.TryParse(test.Line), test.Row, "Data does not match");
        }
    }

    Y_UNIT_TEST(Skip) {
        const std::vector<TString> tests{
            "v=1\taction=not_create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
            "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
            "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",

            "v=1\taction=create\ttarget=not_token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
            "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=1\t"
            "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",

            "v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=not_password\tuid=4073655694\ttoken_id=26452866\t"
            "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\tpassword_passed=0\t"
            "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",

            "v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=not_password\tuid=4073655694\ttoken_id=26452866\t"
            "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=0\t"
            "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",

            "v=1\taction=create\ttarget=token\ttoken_type=normal\tgrant_type=password\tuid=4073655694\ttoken_id=26452866\t"
            "client_id=1898303434af426282ce7d45c89629c1\tscopes=oauth:grant_xtoken\thas_alias=1\tpassword_passed=1\t"
            "user_ip=77.79.1.11\tconsumer_ip=2a02:6b8:c0b:6815:0:4eeb:7496:0\tuser_agent=Apache-HttpClient/4.5.12 (Java/1.8.0_221)\ttimestamp=1622707526.080980",
        };

        TOAuthParserTest p;
        for (const auto& test : tests) {
            std::optional<TOAuthRow> row;
            UNIT_ASSERT_NO_EXCEPTION(row = p.TryParse(test));
            UNIT_ASSERT(!row.has_value());
        }
    }
}
