#include <passport/infra/daemons/lbchdb/src/parsers/restore_parser.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NCrypto;
using namespace NPassport::NLbchdb::NParser;
using namespace NPassport::NLb;

Y_UNIT_TEST_SUITE(RestoreParser) {
    class TRestoreParserTest: public TRestoreParser {
    public:
        explicit TRestoreParserTest(const TDecryptor& decryptor)
            : TRestoreParser(decryptor)
        {
        }

        using TRestoreParser::TryParse;
    };

    class TRestoreParserHolder {
    public:
        TRestoreParserHolder()
            : Decryptor(std::make_unique<TSimpleKeyRing>(TSimpleKeyRing::TKeys{
                  {666, NUtils::Base64ToBin("S0VZWUtFWVlLRVlZS0VZWUtFWVlLRVlZS0VZWUtFWVk=")},
              }))
            , Parser(Decryptor)
        {
        }

        TDecryptor Decryptor;
        TRestoreParserTest Parser;
    };

    Y_UNIT_TEST(ParseBadTData) {
        struct TTestCase {
            TString Line;
            TString Exception;
        };

        const std::vector<TTestCase> tests{
            {"line", "Invalid field count in entry"},
            {"2 a 2021-05-17T18:34:21.741044+03 1 b DEFINITELY_ENCRYPTED_JSON", "Decryptor: invalid format of encrypted string"},
            {"1 a 2021-05-17T18:34:21.741044+03 1 b {\"a:1}", "Invalid JSON file"},
            {"1 a 2021-05-17T18:60:21.74104403 1 b {\"a\":1}", "Invalid rfctime"},
            {"3 a 2021-05-17T18:34:21.741044+03 1 b {\"a\":1}", "Invalid version"},
            {"a a 2021-05-17T18:34:21.741044+03 1 b {\"a\":1}", "Invalid version"},
            {"1 a 2021-05-17T18:34:21.741044+03 -1 b {\"a\":1}", "Invalid UID"},
            {"1 a 2021-05-17T18:34:21.741044+03 9999999999999999999999999999999999 b {\"a\":1}", "Invalid UID"},
            {"1 a 2021-05-17T18:34:21.741044+03 a b {\"a\":1}", "Invalid UID"},
        };

        TRestoreParserHolder p;
        for (const auto& test : tests) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Parser.TryParse(test.Line), yexception, test.Exception);
        }
    }

    Y_UNIT_TEST(ParseGoodTData) {
        struct TTestCase {
            TString Line;
            TRestoreRow Row;
        };

        std::vector<TTestCase> tests;

        TTestCase tc = {
            .Line = "2 restore_semi_auto_request1 2019-05-17T18:34:21.741044+03 10 47,7871,1621265661.73,4072338442,2a10ca0e91419dc8f5c8ebd36e7b2fc947 gNMH92uXi1kjCNUc:uT4=:key_number=666:8yLlAyi2Fqk8TzzXd+ZACg==",
            .Row = {
                .Action = "restore_semi_auto_request1",
                .DataJson = "{}",
                .RestoreId = "47,7871,1621265661.73,4072338442,2a10ca0e91419dc8f5c8ebd36e7b2fc947",
            },
        };
        tc.Row.Timestamp = TInstant::ParseIso8601("2019-05-17T18:34:21.741044+03");
        tc.Row.Uid = 10;
        tests.push_back(std::move(tc));

        tc = {
            .Line = "2 restore_semi_auto_request2 2020-05-17T18:34:21.741044+03 11 47,7867,1621253151.66,4072325468,cb5376691bdcca9b0785f852c9bc3f9f47 6ThMEri8ZUsiF4Gc:asIQnfAyrZl/ZA==:key_number=666:sGUNScY9a/vFyKJNlTrCEg==",
            .Row = {
                .Action = "restore_semi_auto_request2",
                .DataJson = "{\"a\": 234}",
                .RestoreId = "47,7867,1621253151.66,4072325468,cb5376691bdcca9b0785f852c9bc3f9f47",
            }};
        tc.Row.Timestamp = TInstant::ParseIso8601("2020-05-17T18:34:21.741044+03");
        tc.Row.Uid = 11;
        tests.push_back(std::move(tc));

        tc = {
            .Line = "1 restore_semi_auto_request3 2021-05-17T18:34:21.741044+03 12 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 {\"o\":1}",
            .Row = {
                .Action = "restore_semi_auto_request3",
                .DataJson = "{\"o\":1}",
                .RestoreId = "47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747",
            },
        };
        tc.Row.Timestamp = TInstant::ParseIso8601("2021-05-17T18:34:21.741044+03");
        tc.Row.Uid = 12;
        tests.push_back(std::move(tc));

        TRestoreParserHolder p;

        for (const auto& test : tests) {
            UNIT_ASSERT_VALUES_EQUAL_C(p.Parser.TryParse(test.Line), test.Row, "Data does not match");
        }
    }

    Y_UNIT_TEST(CompareOperator) {
        TRestoreRow r1{
            .Action = "def",
            .DataJson = "abc",
            .RestoreId = "ghi",
        };

        TRestoreRow r2{
            .Action = "def",
            .DataJson = "abc",
            .RestoreId = "ghi",
        };

        UNIT_ASSERT_VALUES_EQUAL(r1, r2);
        r2.RestoreId = "xyz";
        UNIT_ASSERT_VALUES_UNEQUAL(r1, r2);
        r1.RestoreId = "xyz";
        r1.Timestamp = TInstant::Now();
        r2.Timestamp = r1.Timestamp;
        UNIT_ASSERT_VALUES_EQUAL(r1, r2);
        r2.Timestamp += TDuration::Seconds(2);
        UNIT_ASSERT_VALUES_UNEQUAL(r1, r2);
    }
}
