#include <passport/infra/daemons/lbchdb/src/crypto/decryptor.h>
#include <passport/infra/daemons/lbchdb/src/parsers/yasms_private.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NCrypto;
using namespace NPassport::NLbchdb::NParser;

Y_UNIT_TEST_SUITE(YasmsPrivateParser) {
    Y_UNIT_TEST(tryParse) {
        TDecryptor d(std::make_unique<TSimpleKeyRing>(TSimpleKeyRing::TKeys{
            {618, NUtils::Base64ToBin("S2wnG2nRMHJHbNIiEPjjD5r/vMSbFRd/cAk683DTETo=")},
        }));
        TYasmsPrivateRow row;

        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("", d),
                                       yexception,
                                       "line should start with 'tskv'");
        UNIT_ASSERT(!row.TryParse("tskv\terror=kek\tsms=1\tnumber=+88005553535", d));
        UNIT_ASSERT(!row.TryParse("tskv\tnumber=+88005553535", d));
        UNIT_ASSERT(!row.TryParse("tskv\tsms=1", d));
        UNIT_ASSERT(!row.TryParse("tskv\tsms=1\tnumber=+99999999999999999999\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005", d));

        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=+88005553535\taction=some_action\tunixtimef=100500.005", d),
                                       yexception,
                                       "empty field 'GlobalSmsId'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\tunixtimef=100500.005", d),
                                       yexception,
                                       "empty field 'Action'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action", d),
                                       yexception,
                                       "empty field 'Timestamp'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=kek", d),
                                       yexception,
                                       "invalid unixtime: 'kek'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005", d),
                                       yexception,
                                       "number missing prefix '+': '88005553535'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005\tuid=abc", d),
                                       yexception,
                                       "invalid number in 'uid': 'abc'");

        UNIT_ASSERT(row.TryParse("tskv\ttskv_format=yasms-log\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005\tunixtime=129666\tkek=lol", d));
        UNIT_ASSERT_VALUES_EQUAL("some_id", row.GlobalSmsId);
        UNIT_ASSERT_VALUES_EQUAL("some_action", row.Action);
        UNIT_ASSERT_VALUES_EQUAL(TInstant::ParseIso8601("1970-01-02T03:55:00.005000Z"), row.Timestamp);
        UNIT_ASSERT_VALUES_EQUAL("+88005553535", row.Number);
        UNIT_ASSERT_VALUES_EQUAL(88005553535, row.Phone);
        UNIT_ASSERT_VALUES_EQUAL("", row.Text);
        UNIT_ASSERT_VALUES_EQUAL(1, row.ExtraFields().size());
        UNIT_ASSERT(row.ExtraFields().contains(("kek")));
        UNIT_ASSERT_VALUES_EQUAL("lol", row.ExtraFields().find("kek")->second);

        UNIT_ASSERT(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005\tuid=123\tencryptedtext=b0z9A+1+8BvDbAiB:j63kcG/xsTVMAdnbg63rKLKHbTvtGlrcEyeoNYAaJkgMDNZQ06WfSOPEUxWKWdg/mpoYtV9bRcBJIsXF2FJzx7avezM=:key_number=618:sOtr+3Ez4kpi+pbDWBM2eQ==", d));
        UNIT_ASSERT_VALUES_EQUAL("some_id", row.GlobalSmsId);
        UNIT_ASSERT_VALUES_EQUAL("some_action", row.Action);
        UNIT_ASSERT_VALUES_EQUAL(TInstant::ParseIso8601("1970-01-02T03:55:00.005000Z"), row.Timestamp);
        UNIT_ASSERT_VALUES_EQUAL("+88005553535", row.Number);
        UNIT_ASSERT_VALUES_EQUAL(88005553535, row.Phone);
        UNIT_ASSERT_VALUES_EQUAL("Your confirmation code is 540870. Please enter it in the text field.", row.Text);
        UNIT_ASSERT_VALUES_EQUAL(0, row.ExtraFields().size());

        // bad encryptedtext
        UNIT_ASSERT(row.TryParse("tskv\tsms=1\tnumber=+88005553535\tglobal_smsid=some_id\taction=some_action\tunixtimef=100500.005\tuid=123\tencryptedtext=kek", d));
        UNIT_ASSERT_VALUES_EQUAL("some_id", row.GlobalSmsId);
        UNIT_ASSERT_VALUES_EQUAL("some_action", row.Action);
        UNIT_ASSERT_VALUES_EQUAL(TInstant::ParseIso8601("1970-01-02T03:55:00.005000Z"), row.Timestamp);
        UNIT_ASSERT_VALUES_EQUAL("+88005553535", row.Number);
        UNIT_ASSERT_VALUES_EQUAL(88005553535, row.Phone);
        UNIT_ASSERT_VALUES_EQUAL("", row.Text);
        UNIT_ASSERT_VALUES_EQUAL(0, row.ExtraFields().size());
    }
}
