#include "common.h"

#include <passport/infra/daemons/lbchdb/src/processor.h>
#include <passport/infra/daemons/lbchdb/src/sender_unsubsrcibe_lists.h>
#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/auths.h>
#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/failed_auths.h>
#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/last_auth.h>
#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/succed_auths.h>

#include <passport/infra/libs/cpp/utils/log/logger.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NCrypto;

Y_UNIT_TEST_SUITE(Proccessor) {
    static NLb::TBadLineLoggers<TLogTypeTraits> CreateLogger(const TString& file = "/dev/null",
                                                             ELogType type = ELogType::BlackboxAuth) {
        NLb::TBadLineLoggers<TLogTypeTraits> res;
        res.Init(file, {TLogTypeTraits::ToString(type)});
        return res;
    }

    static TKeyRingPtr CreateKeyring() {
        return std::make_unique<TSimpleKeyRing>(
            TSimpleKeyRing::TKeys{
                {666, NUtils::Base64ToBin("S0VZWUtFWVlLRVlZS0VZWUtFWVlLRVlZS0VZWUtFWVk=")},
            });
    }

    static TEncryptor::TKeyRingMap CreateKeyringMap() {
        TEncryptor::TKeyRingMap keyRingMap;
        keyRingMap.try_emplace(TEncryptor::EKeyRing::Restore, CreateKeyring(), 666);
        return keyRingMap;
    }

    struct TProc {
        TProc(NLb::TBadLineLoggers<TLogTypeTraits>&& badLog = CreateLogger())
            : P(Geobase,
                Ipreg,
                Uatraits,
                {
                    .IsLastauthToHbaseEnabled = true,
                    .IsMailToHbaseEnabled = true,
                    .AuthSamplingPeriod = 300,
                },
                std::move(badLog),
                CreateKeyring(),
                CreateKeyringMap(),
                nullptr)
        {
            Geobase.As = "some as";
            Geobase.RegionId = 42;

            Ipreg.Info = IIpReg::TIpInfo{.IsYandex = true, .IsUser = false};

            Uatraits.Data = {IUaTraits::TData::TMap{
                {"BrowserName", "kek"},
                {"BrowserVersion", "ver1"},
                {"OSName", "lubuntu"},
                {"OSFamily", "lol"},
                {"OSVersion", "1.0.1e"},
            }};
        }

        TTestGeobase Geobase;
        TTestIpReg Ipreg;
        TTestUaTraits Uatraits;
        TProcessor P;
    };

    static NLb::TData ToData(const TString& line) {
        NLb::TData res;
        NLb::TTopicData td;
        td.Data.push_back(NLb::TChunk{
            .Data = line,
        });
        res.Messages.push_back(std::move(td));

        return res;
    }

    static NLb::TDataSet<ELogType>& ToDataSet(NLb::TDataSet<ELogType>& res, NLb::TData data) {
        res.clear();

        res.push_back(NLb::TDataWithType<ELogType>{
            .Data = std::move(data),
            .Type = ELogType::BlackboxAuth,
        });

        return res;
    }

    Y_UNIT_TEST(procOAuthSuccess1) {
        TProc p;

        const TString line = "1 2020-04-23T01:37:38.751992+03 E4 bb 99276884 - - oauthcheck successful clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all; 5.16.114.159 - - - - -\n";

        NLb::TDataSet<ELogType> dataset;
        TQueries res = p.P.Proc(ToDataSet(dataset, ToData(line)));
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());

        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = "612_99276884_9223372035267180749.248008_oauthcheck/1785987593",
                .Params = {
                    {"c:host_id", "228"},
                    {"c:client_name", "bb"},
                    {"c:type", "oauthcheck"},
                    {"c:status", "successful"},
                    {"c:comment", "clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all;"},
                    {"c:user_ip", "5.16.114.159"},
                    {"c:ip.geoid", "42"},
                    {"c:ip.as_list", "some as"},
                    {"c:ip.is_yandex", "1"},
                },
            }));
        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = "99276884",
                .Params = {
                    {"c:lastauth", "1587595058.75"},
                    {"c:type", "oauthcheck"},
                },
            }));
    }

    Y_UNIT_TEST(procOAuthSuccess2) {
        TProc p;

        const TString line = "1 2020-04-23T01:37:38.751992+03 E4 bb 99276884 - - oauthcheck successful clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all; 5.16.114.159 - - - - -\n";

        p.Ipreg.Info.IsYandex = false;

        NLb::TDataSet<ELogType> dataset;
        TQueries res = p.P.Proc(ToDataSet(dataset, ToData(line)));
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());

        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = "612_99276884_9223372035267180749.248008_oauthcheck/1785987593",
                .Params = {
                    {"c:host_id", "228"},
                    {"c:client_name", "bb"},
                    {"c:type", "oauthcheck"},
                    {"c:status", "successful"},
                    {"c:comment", "clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all;"},
                    {"c:user_ip", "5.16.114.159"},
                    {"c:ip.geoid", "42"},
                    {"c:ip.as_list", "some as"},
                },
            }));
        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = "99276884",
                .Params = {
                    {"c:lastauth", "1587595058.75"},
                    {"c:type", "oauthcheck"},
                },
            }));
        auto decode = [](TStringBuf b) {
            return NUtils::Base64ToBin(b);
        };
        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = decode("ODg0AAAAAAXq2FT/+UVZBRByny5vYXV0aGNoZWNr"),
                .Params = {
                    {decode("aToKCm9hdXRoY2hlY2sSFAoQAAAAAAAAAAAAAP//BRBynxAqKmQKIDY1NjQ0MzBhYmI2ZDQ0ZWE4YzA0MmJjMjdlOTZmOTU3EgoxNzg1OTg3NTkzGiBjYjM0MzUwZDlmNzMyMDg1MjQ2OGQ4YjliMWI3OWQ2OSIGSHVhd2VpKgptb2JpbGU6YWxs"), "1:1587595058.75"},
                },
            }));
    }

    Y_UNIT_TEST(procOAuthFail) {
        TProc p;

        NLb::TData data = ToData(
            "1 2020-04-23T01:37:44.608387+03 E4 bb 76461395 ol.medwedewa2009@yandex.ru 2 oauthcreate failed `ip_short within minute: 22/10;ssl=1;` 88.200.214.141 - - - - `com.yandex.mobile.auth.sdk/6.4.1.604010048 (samsung SM-N950F; Android 4.4.2),com.yandex.mobile.auth.sdk/6.9.1.609010144 (samsung SM-N9005; Android 5.1.1)`\n");
        NLb::TDataSet<ELogType> dataset;
        TQueries res = p.P.Proc(ToDataSet(dataset, std::move(data)));
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());

        UNIT_ASSERT_VALUES_EQUAL(
            res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].at(0),
            NHbase::TQuery({
                .Type = NHbase::TQuery::EType::Put,
                .Row = "76461395_9223372035267180743.391613_228",
                .Params = {
                    {"c:host_id", "228"},
                    {"c:client_name", "bb"},
                    {"c:login", "ol.medwedewa2009@yandex.ru"},
                    {"c:sid", "2"},
                    {"c:type", "oauthcreate"},
                    {"c:status", "failed"},
                    {"c:comment", "ip_short within minute: 22/10;ssl=1;"},
                    {"c:user_ip", "88.200.214.141"},
                    {"c:ip.geoid", "42"},
                    {"c:ip.as_list", "some as"},
                    {"c:ip.is_yandex", "1"},
                    {"c:browser.name", "kek"},
                    {"c:browser.version", "ver1"},
                    {"c:os.name", "lubuntu"},
                    {"c:os.family", "lol"},
                    {"c:os.version", "1.0.1e"},
                },
            }));
    }

    Y_UNIT_TEST(procDataSet) {
        NLb::TDataSet<ELogType> dataset;
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("1 2020-04-23T01:37:44.608387+03 E4 bb 76461395 ` ol.medwedewa2009@yandex.ru` 2 oauthcreate failed `ip_short within minute: 22/10;ssl=1;` 88.200.214.141 - - - - `com.yandex.mobile.auth.sdk/6.4.1.604010048 (samsung SM-N950F; Android 4.4.2),com.yandex.mobile.auth.sdk/6.9.1.609010144 (samsung SM-N9005; Android 5.1.1)`\n"),
            .Type = ELogType::BlackboxAuth,
        });

        TProc p;
        TQueries res = p.P.Proc(dataset);
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());

        dataset.back().Data.Messages.back().Data.push_back(
            NLb::TChunk{
                .Data = "1 2020-04-23T02:37:38.751992+03 E4 bb 99276884 - - oauthcheck successful clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all; 5.16.114.159 - - - - -\n",
            });
        res = p.P.Proc(dataset);
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());

        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("1 2020-04-23T01:37:38.751992+03 E4 bb 99276884 - - oauthcheck successful clid=6564430abb6d44ea8c042bc27e96f957;tokid=1785987593;devid=cb34350d9f7320852468d8b9b1b79d69;devnm=Huawei;scope=mobile:all; 5.16.114.159 - - - - -\n"),
            .Type = ELogType::BlackboxAuth,
        });
        res = p.P.Proc(dataset);
        UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(1, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(2, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
        UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());
    }

    Y_UNIT_TEST(procBadLog) {
        const TString file = "./tmp.log";
        NFs::Remove(file);

        {
            TProc p(CreateLogger(file));

            NLb::TDataSet<ELogType> dataset;
            // invalid uid
            dataset.push_back(NLb::TDataWithType<ELogType>{
                .Data = ToData("1 2020-04-23T01:37:44.608387+03 E4 bb 7646aaaaaaaaaa1395 ` ol.medwedewa2009@yandex.ru` 2 oauthcreate failed `ip_short within minute: 22/10;ssl=1;` 88.200.214.141 - - - - `com.yandex.mobile.auth.sdk/6.4.1.604010048 (samsung SM-N950F; Android 4.4.2)`\n"),
                .Type = ELogType::BlackboxAuth,
            });
            TQueries res = p.P.Proc(dataset);
            UNIT_ASSERT_VALUES_EQUAL(4, res.Hbase.ByTable.size());
            UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TAuthQuery::TABLE_NAME].size());
            UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TFailedAuthQuery::TABLE_NAME].size());
            UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TLastAuth::TABLE_NAME].size());
            UNIT_ASSERT_VALUES_EQUAL(0, res.Hbase.ByTable[NHBaseConv::NAuth::TSuccessAuth::TABLE_NAME].size());
        }
        UNIT_ASSERT_VALUES_EQUAL("1 2020-04-23T01:37:44.608387+03 E4 bb 7646aaaaaaaaaa1395 ` ol.medwedewa2009@yandex.ru` 2 oauthcreate failed `ip_short within minute: 22/10;ssl=1;` 88.200.214.141 - - - - `com.yandex.mobile.auth.sdk/6.4.1.604010048 (samsung SM-N950F; Android 4.4.2)`\n",
                                 TFileInput(file).ReadAll());
    }

    Y_UNIT_TEST(stringToLogType) {
        UNIT_ASSERT_VALUES_EQUAL(ELogType::Auth, TLogTypeTraits::FromString("auth"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::BlackboxAuth, TLogTypeTraits::FromString("bb_auth"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::Event, TLogTypeTraits::FromString("event"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::MailUserJournal, TLogTypeTraits::FromString("mail_user_journal"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::OAuth, TLogTypeTraits::FromString("oauth"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::Restore, TLogTypeTraits::FromString("restore"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::SenderDelivery, TLogTypeTraits::FromString("sender_delivery"));
        UNIT_ASSERT_VALUES_EQUAL(ELogType::YasmsPrivate, TLogTypeTraits::FromString("yasms_private"));
    }

    Y_UNIT_TEST(logTypeAsString) {
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::Auth), "auth");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::BlackboxAuth), "bb_auth");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::Event), "event");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::MailUserJournal), "mail_user_journal");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::OAuth), "oauth");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::Restore), "restore");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::SenderDelivery), "sender_delivery");
        UNIT_ASSERT_VALUES_EQUAL(TLogTypeTraits::ToString(ELogType::YasmsPrivate), "yasms_private");
    }

    Y_UNIT_TEST(procRestore) {
        TProc p(CreateLogger("/dev/null", ELogType::Restore));

        NHbase::TQueryArray queries;

        NHbase::TQuery q{
            .Row = "4072338442_9223372035233510146.258956_-.-",
            .Params = {
                {"r:restore_id", "47,7871,1621265661.73,4072338442,2a10ca0e91419dc8f5c8ebd36e7b2fc947"},
                {"r:action", "restore_semi_auto_request"},
                {"r:data_json", R"({"a":1,"b":5})"},
            },
        };
        queries.push_back(std::move(q));

        q = {
            .Row = "4072325468_9223372035233522656.329798_-.-",
            .Params = {
                {"r:restore_id", "47,7867,1621253151.66,4072325468,cb5376691bdcca9b0785f852c9bc3f9f47"},
                {"r:action", "restore_semi_auto_request"},
                {"r:data_json", R"({"def": "asdf"})"},
            },
        };
        queries.push_back(std::move(q));

        q = {
            .Row = "4071809012_9223372035234300362.285664_-.-",
            .Params = {
                {"r:restore_id", "47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747"},
                {"r:action", "restore_semi_auto_request"},
                {"r:data_json", R"({"wref":{"asdf":[1,2,"fdas",{"fewr":432}]}})"},
            },
        };
        queries.push_back(std::move(q));

        NLb::TDataSet<ELogType> dataset;
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2 restore_semi_auto_request 2021-05-17T18:34:21.741044+03 4072338442 47,7871,1621265661.73,4072338442,2a10ca0e91419dc8f5c8ebd36e7b2fc947 KTjXDjAiGX0+UgYu:g4CWNx/ln/Qe87sOVA==:key_number=666:4uZ/X3wcidoBW8jFV9uFCg==\n"),
            .Type = ELogType::Restore,
        });

        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2 restore_semi_auto_request 2021-05-17T15:05:51.670202+03 4072325468 47,7867,1621253151.66,4072325468,cb5376691bdcca9b0785f852c9bc3f9f47 VVwYMmPd7NFHXk+j:sOmMkoBZBGXmTZeelmoL:key_number=666:LwZ0ELMWqEWSwP4yEFE+QA==\n"),
            .Type = ELogType::Restore,
        });

        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("1 restore_semi_auto_request 2021-05-08T15:04:05.714336+03 4071809012 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 {\"wref\":{\"asdf\":[1,2,\"fdas\",{\"fewr\":432}]}}\n"),
            .Type = ELogType::Restore,
        });

        // Malformed encrypted data
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2 restore_semi_auto_request 2021-05-08T15:04:05.714336+03 4071809012 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 NOT_ENCRYPTED\n"),
            .Type = ELogType::Restore,
        });

        // Incorrect number of csv rows
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2restore_semi_auto_request 2021-05-08T15:04:05.714336+03 4071809012 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 VVwYMmPd7NFHXk+j:sOmMkoBZBGXmTZeelmoL:key_number=666:LwZ0ELMWqEWSwP4yEFE+QA==\n"),
            .Type = ELogType::Restore,
        });

        // Invalid json
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2 restore_semi_auto_request 2021-05-08T15:04:05.714336+03 4071809012 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 /SDHVuqR1YEdJZFf:qQCyo38pisd8rE0d:key_number=666:ZpRU0ldhy0KXnG4T5ojsMA==\n"),
            .Type = ELogType::Restore,
        });

        // Invalid timestamp
        dataset.push_back(NLb::TDataWithType<ELogType>{
            .Data = ToData("2 restore_semi_auto_request 2021-05-08T15:04:05.714336+03as 4071809012 47,2176,1620475445.7,4071809012,c27a1d0c2bf9817a70e17053ae6a481747 VVwYMmPd7NFHXk+j:sOmMkoBZBGXmTZeelmoL:key_number=666:LwZ0ELMWqEWSwP4yEFE+QA==\n"),
            .Type = ELogType::Restore,
        });

        TQueries qs = p.P.Proc(dataset);

        UNIT_ASSERT(qs.Hbase.ByTable.contains("events"));
        UNIT_ASSERT_VALUES_EQUAL(qs.Hbase.ByTable.at("events").size(), queries.size());
        for (size_t i = 0; i < queries.size(); ++i) {
            UNIT_ASSERT_VALUES_EQUAL(qs.Hbase.ByTable.at("events")[i], queries[i]);
        }
    }
}
