#include <passport/infra/daemons/lbchdb/src/extenders/auth_extender.h>
#include <passport/infra/daemons/lbchdb/src/sampler/sampler.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NSampler;

Y_UNIT_TEST_SUITE(ConvertSampler) {
    Y_UNIT_TEST(internalCommon) {
        TInternalSamplerStorage storage({.EntryLimit = 2});

        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Added, storage.TryAdd("#1"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#1"));

        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Added, storage.TryAdd("#2"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#2"));

        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#1"));

        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Added, storage.TryAdd("#3"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#3"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Added, storage.TryAdd("#2"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#3"));
        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Found, storage.TryAdd("#2"));

        UNIT_ASSERT_VALUES_EQUAL(TInternalSamplerStorage::EResult::Added, storage.TryAdd("#1"));
    }

    static NExtend::TAuthExtendedEntry CreateEntry(ui64 uid) {
        NParser::TAuthRow r;
        r.Uid = uid;
        return {.Row = std::move(r)};
    }

    static void SortRes(TBbAuthSampleBulk::TResult& res) {
        std::sort(res.begin(), res.end(), [](const auto& l, const auto& r) {
            return l.Row.Uid < r.Row.Uid;
        });
    }

    Y_UNIT_TEST(samplerCommon) {
        TBbAuthSampler sampler({});

        {
            TBbAuthSampleBulk b(sampler);
            b.Add("#1", CreateEntry(100));
            b.Add("#2", CreateEntry(200));
            b.Add("#1", CreateEntry(100));
            auto res = b.Calculate();
            SortRes(res);

            UNIT_ASSERT_VALUES_EQUAL(2, res.size());
            UNIT_ASSERT_VALUES_EQUAL(100, res[0].Row.Uid);
            UNIT_ASSERT_VALUES_EQUAL(200, res[1].Row.Uid);
        }

        {
            TBbAuthSampleBulk b(sampler);
            b.Add("#1", CreateEntry(100));
            b.Add("#3", CreateEntry(300));
            auto res = b.Calculate();

            UNIT_ASSERT_VALUES_EQUAL(1, res.size());
            UNIT_ASSERT_VALUES_EQUAL(300, res[0].Row.Uid);
        }
    }

    Y_UNIT_TEST(createRequest) {
        TBbAuthSampler sampler({.KolmogorSpace = "some_space"});
        TBbAuthSampleBulk b(sampler);

        b.Add("key#1", {});

        UNIT_ASSERT_VALUES_EQUAL(R"({"some_space":[{"keys":["key#1"]}]})",
                                 b.CreateKolmoRequest());
    }

    Y_UNIT_TEST(parseResponse) {
        auto test = [](auto func) {
            TBbAuthSampler sampler({.KolmogorSpace = "some_space"});
            TBbAuthSampleBulk b(sampler);

            rapidjson::Document doc;
            const rapidjson::Value* space = nullptr;
            func(b, doc, space);
        };

        test([](TBbAuthSampleBulk& b, auto& doc, auto space) {
            UNIT_ASSERT_NO_EXCEPTION(
                b.ParseResponse(R"({"some_space":{"key#1":{"value":2}}})",
                                doc,
                                space));
            std::optional<ui64> val = TBbAuthSampleBulk::GetCounterFromResponse(space, "key#1");
            UNIT_ASSERT(val);
            UNIT_ASSERT_VALUES_EQUAL(*val, 2);
        });

        test([](TBbAuthSampleBulk& b, auto& doc, auto space) {
            UNIT_ASSERT_NO_EXCEPTION(
                b.ParseResponse(R"({"some_space":{"key#1":{"value":1}}})",
                                doc,
                                space));
            std::optional<ui64> val = TBbAuthSampleBulk::GetCounterFromResponse(space, "key#1");
            UNIT_ASSERT(val);
            UNIT_ASSERT_VALUES_EQUAL(*val, 1);
        });

        test([](TBbAuthSampleBulk& b, auto& doc, auto space) {
            UNIT_ASSERT_NO_EXCEPTION(
                b.ParseResponse(R"({"some_space":{"key#1":["value", 2]}})",
                                doc,
                                space));
            UNIT_ASSERT(!b.GetCounterFromResponse(space, "key#1"));
        });

        test([](TBbAuthSampleBulk& b, auto& doc, auto space) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                b.ParseResponse(R"([])",
                                doc,
                                space),
                yexception,
                "invalid json from kolmogor: []");
        });

        test([](TBbAuthSampleBulk& b, auto& doc, auto space) {
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                b.ParseResponse(R"({"some_other_space":{"key#1":{"value":2}}})",
                                doc,
                                space),
                yexception,
                R"(missing space 'some_space' in response: {"some_other_space":{"key#1":{"value":2}}})");
        });
    }

    Y_UNIT_TEST(wasFoundInKolmogor) {
        UNIT_ASSERT(!TBbAuthSampleBulk::WasFoundInKolmogor({}));
        UNIT_ASSERT(!TBbAuthSampleBulk::WasFoundInKolmogor(0));
        UNIT_ASSERT(!TBbAuthSampleBulk::WasFoundInKolmogor(1));
        UNIT_ASSERT(TBbAuthSampleBulk::WasFoundInKolmogor(2));
        UNIT_ASSERT(TBbAuthSampleBulk::WasFoundInKolmogor(3));
    }
}
