#include <passport/infra/daemons/lbchdb/ut/common.h>

#include <passport/infra/daemons/lbchdb/src/yt_converters/push_converter.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbchdb;
using namespace NPassport::NLbchdb::NYtConv;
using namespace NPassport::NLbchdb::NYtConv::NPush;

Y_UNIT_TEST_SUITE(ConvertYtPushTable) {
    Y_UNIT_TEST(buildData) {
        TPushEntry pushEntry;

        pushEntry = TEntryBuilder(NParser::TPushRow{}).BuildPushEntry();
        UNIT_ASSERT_VALUES_EQUAL(ToYson({}),
                                 pushEntry.Data);

        pushEntry = TEntryBuilder(NParser::TPushRow{
                                      .Status = "status",
                                  })
                        .BuildPushEntry();
        UNIT_ASSERT_VALUES_EQUAL(ToYson({{"status", "status"}}),
                                 pushEntry.Data);

        pushEntry = TEntryBuilder(NParser::TPushRow{
                                      .Context = "some context",
                                  })
                        .BuildPushEntry();
        UNIT_ASSERT_VALUES_EQUAL(ToYson({{"context", "some context"}}),
                                 pushEntry.Data);

        pushEntry = TEntryBuilder(NParser::TPushRow{
                                      .Status = "status",
                                      .Context = "some context",
                                  })
                        .BuildPushEntry();
        UNIT_ASSERT_VALUES_EQUAL(ToYson({{"context", "some context"}, {"status", "status"}}),
                                 pushEntry.Data);
    }

    Y_UNIT_TEST(buildEntry) {
        NParser::TPushRow row{
            .Uid = 666,
            .Unixtime = TInstant::Seconds(16000000),

            .PushId = "a:b:c",
            .SubscriptionId = "mob:100500",
            .AppId = "taxi",
            .DeviceId = "device999",

            .PushService = "foo",
            .PushEvent = "bar",

            .Details = "some_details",

            .Status = "some status",
            .Context = "some context",
        };

        TEntryBuilder builder(row);

        TPushEntry pushEntry = builder.BuildPushEntry();
        UNIT_ASSERT_VALUES_EQUAL(666, pushEntry.Uid);
        UNIT_ASSERT_VALUES_EQUAL(16000000, pushEntry.Unixtime);
        UNIT_ASSERT_VALUES_EQUAL("a:b:c", pushEntry.Ids->PushId);
        UNIT_ASSERT_VALUES_EQUAL("mob:100500", pushEntry.Ids->SubscriptionId);
        UNIT_ASSERT_VALUES_EQUAL("taxi", pushEntry.Ids->AppId);
        UNIT_ASSERT_VALUES_EQUAL("device999", pushEntry.Ids->DeviceId);
        UNIT_ASSERT_VALUES_EQUAL(ToYson({{"context", "some context"}, {"status", "some status"}, {"push_service", "foo"}, {"push_event", "bar"}, {"details", "some_details"}}), pushEntry.Data);

        TPushIndxEntry pushIndxEntry = builder.BuildPushIndxEntry();
        UNIT_ASSERT_VALUES_EQUAL(666, pushIndxEntry.Uid);
        UNIT_ASSERT_VALUES_EQUAL(9223372036838775807, pushIndxEntry.ReversedUnixtime);
        UNIT_ASSERT_VALUES_EQUAL("a:b:c", pushIndxEntry.Ids->PushId);
        UNIT_ASSERT_VALUES_EQUAL("mob:100500", pushIndxEntry.Ids->SubscriptionId);
        UNIT_ASSERT_VALUES_EQUAL("taxi", pushIndxEntry.Ids->AppId);
        UNIT_ASSERT_VALUES_EQUAL("device999", pushIndxEntry.Ids->DeviceId);
    }

    Y_UNIT_TEST(processConvert) {
        TPushConverter converter;

        converter.Process(
            NParser::TPushRow{
                .Uid = 666,
                .Unixtime = TInstant::Seconds(10000000),

                .PushId = "a:b:c",
                .SubscriptionId = "mob:100500",
                .AppId = "taxi",
                .DeviceId = "device999",

                .PushService = "foo",
                .PushEvent = "bar",

                .Details = "some_details",

                .Status = "some status",
                .Context = "some context",
            },
            TTablesLifeTimeConfig{.PushTtl = TDuration::Days(100000)});

        converter.Process(
            NParser::TPushRow{
                .Uid = 666,
                .Unixtime = TInstant::Seconds(10000500),

                .PushId = "foo",
                .SubscriptionId = "fake"},
            TTablesLifeTimeConfig{.PushTtl = TDuration::Days(100000)});

        converter.Process(
            NParser::TPushRow{
                .Uid = 777,
                .Unixtime = TInstant::Seconds(20000000),

                .PushId = "bar",
                .SubscriptionId = "fake"},
            TTablesLifeTimeConfig{.PushTtl = TDuration::Days(100000)});

        NYt::TWriteQueries<TString> res = converter.Finish();

        UNIT_ASSERT_EQUAL(8, res.ByTable.size());

        TString firstDate = NYt::TMonthlyTable(10000000).TableName();
        TString secondDate = NYt::TMonthlyTable(20000000).TableName();

        auto it = res.ByTable.find("push/" + firstDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(2, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=666. unixtime=10000000. push_id=a:b:c. subscription_id=mob:100500. "
                                 "app_id=taxi. device_id=device999. "
                                 "data={\"context\":\"some context\",\"status\":\"some status\",\"push_service\":\"foo\",\"push_event\":\"bar\",\"details\":\"some_details\"}\n#"
                                 "uid=666. unixtime=10000500. push_id=foo. subscription_id=fake. app_id=. device_id=. "
                                 "data={}\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push-by-uid/" + firstDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(2, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=666. reversed_unixtime=9223372036844775807. push_id=a:b:c. subscription_id=mob:100500. app_id=taxi. device_id=device999\n#"
                                 "uid=666. reversed_unixtime=9223372036844775307. push_id=foo. subscription_id=fake. app_id=. device_id=\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push-by-device-id/" + firstDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(1, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=666. reversed_unixtime=9223372036844775807. push_id=a:b:c. subscription_id=mob:100500. app_id=taxi. device_id=device999\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push-by-app-id/" + firstDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(1, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=666. reversed_unixtime=9223372036844775807. push_id=a:b:c. subscription_id=mob:100500. app_id=taxi. device_id=device999\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push/" + secondDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(1, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=777. unixtime=20000000. push_id=bar. subscription_id=fake. app_id=. device_id=. "
                                 "data={}\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push-by-uid/" + secondDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(1, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("uid=777. reversed_unixtime=9223372036834775807. push_id=bar. subscription_id=fake. app_id=. device_id=\n#",
                                 it->second->DebugString());

        it = res.ByTable.find("push-by-device-id/" + secondDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(0, it->second->size());

        it = res.ByTable.find("push-by-app-id/" + secondDate);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL(0, it->second->size());
    }
}
