package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strconv"
)

func userinfoResponse(uid string) interface{} {
	users := make([]interface{}, 0, 1)
	if uid != "" {
		users = append(
			users,
			map[string]interface{}{
				"uid": map[string]interface{}{
					"value": uid,
				},
			},
		)
	}

	return map[string]interface{}{
		"users": users,
	}
}

func emailBindingsResponse(uids []string) interface{} {
	if uids == nil {
		uids = []string{}
	}

	return map[string]interface{}{
		"uids": uids,
	}
}

func errorResponse(format string, a ...interface{}) interface{} {
	return map[string]interface{}{
		"exception": map[string]interface{}{
			"value": "INVALID_PARAMS",
			"id":    2,
		},
		"error": fmt.Sprintf(format, a...),
	}
}

func BlackboxHandler(w http.ResponseWriter, r *http.Request) {
	if err := r.ParseForm(); err != nil {
		fmt.Println("Failed to parse form: ", err)
	}

	var response interface{}
	switch method := r.FormValue("method"); method {
	case "userinfo":
		uidByLogin := map[string]string{
			"vasya@kek.lol":      "129",
			"babushka@balalaika": "666",
		}
		response = userinfoResponse(uidByLogin[r.FormValue("login")])
	case "email_bindings":
		bindingsByEmail := map[string][]string{
			"jorik@foo.bar":      []string{"17", "100500"},
			"babushka@balalaika": []string{"17", "129", "666"},
		}
		response = emailBindingsResponse(bindingsByEmail[r.FormValue("email")])
	default:
		response = errorResponse("no mock for method '%s'", method)
	}

	jsonResponse, err := json.Marshal(response)
	if err != nil {
		fmt.Println(err)
	}

	w.WriteHeader(http.StatusOK)
	if _, err := w.Write(jsonResponse); err != nil {
		fmt.Println("Failed to write response: ", err)
	}
}

func SimpleHandler(response string) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		_, err := w.Write([]byte(response))
		if err != nil {
			fmt.Printf("Failed to write response: %s", err)
		}
	}
}

func RunServer(port uint16) {
	fmt.Printf("Starting \"blackbox mock\" server at port %d\n", port)

	http.HandleFunc("/ping", SimpleHandler("pong"))
	http.HandleFunc("/blackbox/check_grants", SimpleHandler("{\"grants_status\":\"OK\"}"))
	http.HandleFunc("/blackbox", BlackboxHandler)

	err := http.ListenAndServe(fmt.Sprintf(":%d", port), nil)
	if err != http.ErrServerClosed {
		fmt.Println(err)
	}
}

func main() {
	if len(os.Args) < 2 {
		fmt.Printf("Usage: %s <port>\n", os.Args[0])
		return
	}

	port, err := strconv.ParseUint(os.Args[1], 10, 16)
	if err != nil {
		fmt.Printf("Invalid argument <port>: must be uint, got %s\n", os.Args[1])
		return
	}

	RunServer(uint16(port))
}
