import datetime
import time
import yt.wrapper

PUSH_TABLE_DIR = "//home/push"
PUSH_BY_UID_TABLE_DIR = "//home/push-by-uid"
PUSH_BY_DEVICE_ID_TABLE_DIR = "//home/push-by-device-id"
PUSH_BY_APP_ID_TABLE_DIR = "//home/push-by-app-id"

PUSH_SUBSCRIPTION_TABLE = "//home/push_subscription/push_subscription"

USERS_HISTORY_TABLE_DIR = "//home/users_history"
CORP_USERS_HISTORY_TABLE_DIR = "//home/corp_users_history"

SENDR_TABLE_NAME = "//home/sendr/last_letter"

LASTAUTH_TABLE_NAME = "//home/lastauth/lastauth"
FAILED_AUTHS_TABLE_NAME = "//home/failed_auths/failed_auths"
AUTHS_TABLE_DIR = "//home/auths"

RESTORE_TABLE_NAME = "//home/restore/restore"

YASMS_SMS_HISTORY_TABLE = "//home/yasms/sms_history/sms_history"
YASMS_SMS_HISTORY_BY_PHONE_TABLE = "//home/yasms/sms_history_by_phone/sms_history_by_phone"
YASMS_SMS_HISTORY_BY_UID_TABLE = "//home/yasms/sms_history_by_uid/sms_history_by_uid"


class MonthlyTable:
    def __init__(self, ts=None):
        if ts is None:
            ts = int(time.time())
        self._date = self.__get_month_start_by_ts(ts)

    @staticmethod
    def __get_month_start_by_ts(ts):
        date = datetime.datetime.fromtimestamp(ts)
        return datetime.datetime(year=date.year, month=date.month, day=1)

    def date(self):
        return self._date

    def ts(self):
        return int(self._date.timestamp())

    def name(self):
        return self._date.strftime("%Y-%m")

    def path(self, directory):
        return '/'.join((directory, self.name()))

    def prev(self):
        return MonthlyTable(self.ts() - 1)

    def next(self):
        return MonthlyTable(self.ts() + 60 * 60 * 24 * 31)


def create_dir(dir):
    if yt.wrapper.exists(dir):
        yt.wrapper.remove(dir, recursive=True)

    yt.wrapper.create("map_node", dir, recursive=True)


def create_table(table_name, schema, atomicity="none", **attributes):
    if yt.wrapper.exists(table_name):
        yt.wrapper.remove(table_name)

    yt.wrapper.create(
        "table",
        table_name,
        attributes={
            "dynamic": True,
            "schema": schema,
            "atomicity": atomicity,
            **attributes,
        },
        recursive=True,
    )
    yt.wrapper.mount_table(table_name)


def create_monthly_tables(directory, schema, n=3, ts=None, **attributes):
    create_dir(directory)

    monthly_table = MonthlyTable(ts).next()
    for _ in range(n + 1):
        create_table(monthly_table.path(directory), schema, **attributes)
        monthly_table = monthly_table.prev()


def prepare_yt():
    create_dir(PUSH_TABLE_DIR)
    create_dir(PUSH_BY_UID_TABLE_DIR)
    create_dir(PUSH_BY_DEVICE_ID_TABLE_DIR)
    create_dir(PUSH_BY_APP_ID_TABLE_DIR)

    create_push_subscription_table()

    create_dir(USERS_HISTORY_TABLE_DIR)
    create_dir(CORP_USERS_HISTORY_TABLE_DIR)

    create_sendr_table()

    create_lastauth_table()
    create_auths_tables(failed=True)
    create_dir(AUTHS_TABLE_DIR)

    create_restore_table()

    create_yasms_sms_history_tables()


def create_push_tables(n=3):
    push_schema = [
        {
            "name": "push_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "subscription_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "uid",
            "type": "uint64",
        },
        {
            "name": "device_id",
            "type": "string",
        },
        {
            "name": "app_id",
            "type": "string",
        },
        {
            "name": "unixtime",
            "type": "datetime",
        },
        {
            "name": "data",
            "type": "any",
        },
    ]

    push_by_uid_schema = [
        {
            "name": "uid",
            "type": "uint64",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "reversed_timestamp",
            "type": "int64",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "push_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "subscription_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "dummy",
            "type": "any",
        },
    ]

    push_by_device_id_schema = [
        {
            "name": "device_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "reversed_timestamp",
            "type": "int64",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "push_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "subscription_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "dummy",
            "type": "any",
        },
    ]

    push_by_app_id_schema = [
        {
            "name": "uid",
            "type": "uint64",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "app_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "reversed_timestamp",
            "type": "int64",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "push_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "subscription_id",
            "type": "string",
            "sort_order": "ascending",
            "required": True,
        },
        {
            "name": "dummy",
            "type": "any",
        },
    ]

    ts = int(time.time())

    create_monthly_tables(PUSH_TABLE_DIR, push_schema, n, ts)
    create_monthly_tables(PUSH_BY_UID_TABLE_DIR, push_by_uid_schema, n, ts)
    create_monthly_tables(PUSH_BY_DEVICE_ID_TABLE_DIR, push_by_device_id_schema, n, ts)
    create_monthly_tables(PUSH_BY_APP_ID_TABLE_DIR, push_by_app_id_schema, n, ts)

    return ts


def create_push_subscription_table():
    schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "app_id",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "device_id",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "timestamp",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "count",
            "required": True,
            "type_v3": "uint64",
            "aggregate": "sum",
        },
    ]

    create_table(PUSH_SUBSCRIPTION_TABLE, schema)


def create_users_history_tables(n=3):
    schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "reversed_unixtime",
            "required": True,
            "type_v3": "int64",
            "sort_order": "ascending",
        },
        {
            "name": "operation",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "module",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "data",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "yson"},
        },
    ]

    create_monthly_tables(USERS_HISTORY_TABLE_DIR, schema, n)
    create_monthly_tables(CORP_USERS_HISTORY_TABLE_DIR, schema, n)


def create_sendr_table():
    sendr_schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "unsibscribe_list",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "timestamp",
            "required": True,
            "type_v3": "uint64",
            "aggregate": "max",
        },
    ]

    create_table(SENDR_TABLE_NAME, sendr_schema)


def create_lastauth_table():
    schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "authtype",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "timestamp",
            "required": True,
            "type_v3": "uint64",
            "aggregate": "max",
        },
    ]

    create_table(LASTAUTH_TABLE_NAME, schema)


def create_auths_tables(n=3, failed=False):
    schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "reversed_timestamp",
            "required": True,
            "type_v3": "int64",
            "sort_order": "ascending",
        },
        {
            "name": "unique_part",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "type",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "string"},
        },
        {
            "name": "status",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "string"},
        },
        {
            "name": "client_name",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "string"},
        },
        {
            "name": "data",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "yson"},
        },
    ]

    if failed:
        create_table(FAILED_AUTHS_TABLE_NAME, schema)
    else:
        create_monthly_tables(AUTHS_TABLE_DIR, schema, n)


def create_restore_table():
    schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "reversed_timestamp",
            "required": True,
            "type_v3": "int64",
            "sort_order": "ascending",
        },
        {
            "name": "data",
            "type": "any",
        },
    ]

    create_table(RESTORE_TABLE_NAME, schema)


def create_yasms_sms_history_tables():
    main_schema = [
        {
            "name": "global_sms_id",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "action",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "unixtime",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "int64"},
        },
        {
            "name": "uid",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "uint64"},
        },
        {
            "name": "phone",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "uint64"},
        },
        {
            "name": "data",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "yson"},
        },
    ]

    by_phone_schema = [
        {
            "name": "phone",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "reversed_timestamp",
            "required": True,
            "type_v3": "int64",
            "sort_order": "ascending",
        },
        {
            "name": "global_sms_id",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "action",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "dummy",
            "type": "any",
        },
    ]

    by_uid_schema = [
        {
            "name": "uid",
            "required": True,
            "type_v3": "uint64",
            "sort_order": "ascending",
        },
        {
            "name": "reversed_timestamp",
            "required": True,
            "type_v3": "int64",
            "sort_order": "ascending",
        },
        {
            "name": "global_sms_id",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "action",
            "required": True,
            "type_v3": "string",
            "sort_order": "ascending",
        },
        {
            "name": "dummy",
            "type": "any",
        },
    ]

    create_table(YASMS_SMS_HISTORY_TABLE, main_schema)
    create_table(YASMS_SMS_HISTORY_BY_PHONE_TABLE, by_phone_schema)
    create_table(YASMS_SMS_HISTORY_BY_UID_TABLE, by_uid_schema)
