import brotli
import gzip
import json
import time
import zstandard as zstd

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)


def reverse_timestamp(ts):
    return 0x7FFFFFFFFFFFFFFF - ts


def decompress(data, codec):
    if codec == 'gzip':
        return gzip.decompress(data)
    if codec == 'brotli':
        return brotli.decompress(data)
    if codec == 'zstd':
        return zstd.ZstdDecompressor().decompress(data)

    assert False, "Unknown compression codec: %s" % codec


def decrypt(key, iv, ciphertext, tag, associated_data=None):
    decryptor = Cipher(
        algorithms.AES(key),
        modes.GCM(iv, tag),
        backend=default_backend(),
    ).decryptor()

    if associated_data is not None:
        decryptor.authenticate_additional_data(associated_data)
    return decryptor.update(ciphertext) + decryptor.finalize()


def deep_sorted(item):
    def key_func(x):
        if isinstance(x, dict):
            return sorted((key, key_func(value)) for key, value in x.items())
        return x

    if isinstance(item, dict):
        return {key: deep_sorted(value) for key, value in item.items()}
    if isinstance(item, list):
        return sorted((deep_sorted(x) for x in item), key=key_func)
    if isinstance(item, tuple):
        return tuple(deep_sorted(x) for x in item)
    return item


def pretty_json(value, indent=2, **kwargs):
    return json.dumps(json.loads(value), indent=indent, **kwargs)


def test_with_deadline(func, deadline):
    while True:
        try:
            return func()
        except AssertionError:
            if time.time() > deadline:
                raise
            time.sleep(1)


def test_with_timeout(func, timeout):
    return test_with_deadline(func, time.time() + timeout)
