import base64
import json
import jsondiff
import os
import requests
import time
import yatest.common as yc
import yt.wrapper

from passport.infra.recipes.common import log, start_daemon
from yatest.common import network

import passport.infra.daemons.lbchdb.ut_medium.logbroker as lb

from passport.infra.daemons.lbchdb.ut_medium.helpers import (
    decompress,
    decrypt,
    deep_sorted,
    pretty_json,
    test_with_deadline,
)


class LbchdbFixture:
    logbroker_client_id = "test_client"

    def __init__(self, lb_port=None, yt_proxy=None, timeout=15, check_period__msec=10000):
        self._pm = network.PortManager()

        self.http_port = self._pm.get_tcp_port()
        blackbox_port = self._pm.get_tcp_port()

        output_path = yc.output_path('lbchdb')
        if not os.path.isdir(output_path):
            os.mkdir(output_path)

        logs_dir = os.path.join(output_path, 'logs')
        if not os.path.isdir(logs_dir):
            os.mkdir(logs_dir)

        tvm_secret_path = os.path.join(output_path, 'tvm.secret')
        tvm_cache_dir = os.path.join(output_path, 'tvm_cache')
        oauth_token_path = os.path.join(output_path, 'yt.token')

        geodata_path = yc.work_path('test-geodata.bin')
        layout_path = yc.source_path('passport/infra/daemons/lbchdb/ut_medium/data/layout.json')
        uatraits_path = yc.source_path('passport/infra/daemons/lbchdb/ut_medium/data/browser.xml')

        sender_accounts_path = yc.source_path('passport/infra/daemons/lbchdb/ut_medium/data/sender_accounts.json')

        decryptor_keys_dir = os.path.join(output_path, 'keys')
        encryptor_keys_dir = output_path

        config_path = os.path.join(output_path, 'config.json')

        try:
            with open('./tvmapi.port') as f:
                tvm_port = int(f.read())
                log('tvm port: {}'.format(tvm_port))
        except IOError:
            log('Could not find tvm port file: ./tvmapi.port')

        lbchdb_config = {
            "http_daemon": {
                "listen_address": "localhost",
                "ports": [
                    {
                        "port": self.http_port,
                    },
                ],
            },
            "component": {
                "logger": {
                    "file": os.path.join(logs_dir, 'lbchdb.log'),
                    "print_level": True,
                    "time_format": "_DEFAULT_",
                },
                "dbpool_log": os.path.join(logs_dir, 'dbpool.log'),
                "tvm": {
                    "self_tvm_id": 1000501,
                    "self_secret_file": tvm_secret_path,
                    "disk_cache": tvm_cache_dir,
                    "dsts": [
                        {"alias": "blackbox", "tvm_id": 1000502},
                        {"alias": "kolmogor", "tvm_id": 1000503},
                        {"alias": "localhost", "tvm_id": 1000504},
                    ],
                    "tvm_host": "http://127.0.0.1",
                    "tvm_port": tvm_port,
                },
                "geobase": {
                    "geobase_file": geodata_path,
                    "ipreg_file": layout_path,
                },
                "uatraits": {
                    "file": uatraits_path,
                },
                "logbroker": {
                    "logs": {
                        "auth": {
                            "memory_usage_weight": 10,
                            "topic": lb.TopicType.AUTH,
                        },
                        "bb_auth": {
                            "memory_usage_weight": 45,
                            "topic": lb.TopicType.BB_AUTH,
                        },
                        "mail_user_journal": {
                            "memory_usage_weight": 30,
                            "topic": lb.TopicType.MAIL_USER_JOURNAL,
                        },
                        "push": {
                            "memory_usage_weight": 5,
                            "topic": lb.TopicType.PUSH,
                        },
                        "push_subscription": {
                            "memory_usage_weight": 5,
                            "topic": lb.TopicType.PUSH_SUBSCRIPTION,
                        },
                        "sender_delivery": {
                            "memory_usage_weight": 10,
                            "topic": lb.TopicType.SENDR,
                        },
                        "restore": {
                            "memory_usage_weight": 5,
                            "topic": lb.TopicType.RESTORE,
                        },
                        "yasms_private": {
                            "memory_usage_weight": 5,
                            "topic": lb.TopicType.YASMS_PRIVATE,
                        },
                    },
                    "client_id": self.logbroker_client_id,
                    "server": "localhost",
                    "port": lb_port if lb_port else lb.port(),
                    "inflight_reads": 4,
                    "memory_usage_per_server__mb": 20,
                    "read_max_bytes": 512 * 1024,
                    "read_max_count": 256,
                    "max_uncommitted_total_size": 160 * 1024 * 1024,
                    "reserve_ratio": 0.50,
                    "log_level": 6,
                    "queue_size": 512,
                    "queue_workers": 64,
                    "join_messages": 1,
                },
                "hbase": {
                    "enabled": False,
                },
                "processor": {
                    "enable_lastauth_to_hbase": False,
                    "enable_mail_to_hbase": False,
                    "enable_mail_to_yt": True,
                    "compress_if_more_than": 100,
                    "sender_sample_ratio": 100,
                    "auth_sampling": {
                        "period": 3600,
                        "kolmogor_space": "bb_auth_sampling",
                        "cache_entry_size_limit": 20000000,
                    },
                    "bad_lines_logfile_pattern": os.path.join(logs_dir, 'badlines__%s.log'),
                    "sender_accounts": sender_accounts_path,
                    "decryptor_key_dir_path": decryptor_keys_dir,
                    "encryptor_key_dir_path": encryptor_keys_dir,
                    "restore_default_key": 1,
                    "yasms_private_default_key": 1,
                    "push_subscription_bucket_width": 10000,
                },
                "data_pusher": {
                    "workers": 8,
                    "max_response_time_in_signal__ms": 60000,
                },
                "yt": {
                    "cluster": yt_proxy if yt_proxy else os.environ["YT_PROXY"],
                    "cypress_dir": "//home",
                    "oauth_token_path": oauth_token_path,
                    "log": {
                        "level": 2,
                        "file": os.path.join(logs_dir, 'yt.log'),
                    },
                    "timeout__ms": 10000,
                    "batch_size": 5000,
                    "create_tables": {
                        "users_history": {
                            "desired_tablet_count": 100,
                            "create_timeout__sec": 10,
                            "check_period__msec": check_period__msec,
                            "users_history_table_ttl__sec": 5356800,
                            "corp_users_history_table_ttl__sec": 5356800,
                        },
                        "push": {
                            "desired_tablet_count": 100,
                            "create_timeout__sec": 10,
                            "check_period__msec": check_period__msec,
                            "default_table_ttl__sec": 5356800,
                        },
                        "auths": {
                            "desired_tablet_count": 100,
                            "create_timeout__sec": 10,
                            "check_period__msec": check_period__msec,
                            "auths_table_ttl__sec": 5356800,
                        },
                    },
                },
                "blackbox": {
                    "poolsize": 8,
                    "get_timeout": 3,
                    "connect_timeout": 3000,
                    "query_timeout": 1000,
                    "db_name": "blackbox",
                    "db_driver": "http",
                    "db_port": blackbox_port,
                    "db_host": [{"host": "localhost"}],
                },
                "kolmogor": {
                    "enabled": False,
                },
            },
        }

        with open(tvm_secret_path, 'w') as f:
            f.write('bAicxJVa5uVY7MjDlapthw')
        if not os.path.isdir(tvm_cache_dir):
            os.mkdir(tvm_cache_dir)

        with open(oauth_token_path, 'w') as f:
            f.write('token')

        if not os.path.isdir(decryptor_keys_dir):
            os.mkdir(decryptor_keys_dir)
        with open(os.path.join(decryptor_keys_dir, '666.key'), 'wb') as f:
            f.write(base64.b64decode('d3afIKJ+MB2ZnAMxwsNQN729osL9meb33YP/5wilfCU='))

        def random_key():
            return os.urandom(32)

        def dump_keys(keys):
            return json.dumps({str(id): base64.b64encode(key).decode() for id, key in keys.items()})

        if not os.path.isdir(encryptor_keys_dir):
            os.mkdir(encryptor_keys_dir)

        for key_ring in ("restore", "yasms_private"):
            keys = {1: random_key()}
            with open(os.path.join(encryptor_keys_dir, f'{key_ring}.keys'), 'w') as f:
                # Кладем лишних ключей, что бы жизнь медом не казалась (что бы убедиться, что шифрование идет именно дефолтным)
                f.write(dump_keys({**keys, 0: random_key(), 2: random_key()}))
            setattr(self, f'_{key_ring}_keys', keys)

        with open(config_path, 'w') as f:
            json.dump(lbchdb_config, f, indent=2)

        command = [
            yc.build_path('passport/infra/daemons/lbchdb/ut_medium/blackbox_mock/blackbox_server'),
            str(blackbox_port),
        ]
        self._blackbox_process = start_daemon(command, os.environ.copy(), blackbox_port, 300)

        command = [
            yc.build_path('passport/infra/daemons/lbchdb/daemon/lbchdb'),
            '-c',
            config_path,
        ]
        self._process = start_daemon(command, os.environ.copy(), self.http_port, 300)
        self._deadline = time.time() + timeout

        yt_config = yt.wrapper.default_config.get_config_from_env()
        yt_config["proxy"]["retries"]["backoff"] = {"policy": "constant_time", "constant_time": 500}
        self._ytc = yt.wrapper.YtClient(config=yt_config)

    def stop(self):
        log("Closing lbchdb fixture with %d sec. before deadline" % (self._deadline - time.time()))

        self._pm.release()
        self._process.terminate()
        self._blackbox_process.terminate()

    def healthcheck(self, code=200, resp='0;OK'):
        res = requests.get('http://127.0.0.1:%d/healthcheck' % self.http_port)

        assert res.status_code == code
        assert res.text == resp

    def check_yt_content(self, content, preproc=lambda x: x):
        for table, expected_rows in content.items():
            expected_rows_sorted = deep_sorted(expected_rows)

            def check_table_content():
                assert self._ytc.exists(table), table
                assert self._ytc.get_attribute(table, "dynamic")
                actual_rows = [preproc(r) for r in self._ytc.select_rows("* FROM[{table}]".format(table=table))]
                actual_rows_sorted = deep_sorted(actual_rows)

                assert expected_rows_sorted == actual_rows_sorted, "%s:\nGOT: %s\nEXP: %s\nDIFF: %s\n" % (
                    table,
                    json.dumps(actual_rows_sorted, indent=2),
                    json.dumps(expected_rows_sorted, indent=2),
                    pretty_json(jsondiff.diff(expected_rows_sorted, actual_rows_sorted, dump=True)),
                )

            test_with_deadline(check_table_content, self._deadline)

    @staticmethod
    def decompress(field, codec=None):
        decompressed = decompress(field["value"]._bytes, field["codec"] if codec is None else codec)
        assert len(decompressed) == field["size"]

        return decompressed.decode()

    @staticmethod
    def decrypt(field, keys):
        assert field["v"] == 1

        decrypted = decrypt(keys[field["keyid"]], field["iv"]._bytes, field["text"]._bytes, field["tag"]._bytes)

        if "codec" not in field:
            return decrypted.decode(), False

        decompressed = decompress(decrypted, field["codec"])
        assert len(decompressed) == field["size"]

        return decompressed.decode(), True

    @staticmethod
    def users_history_preproc():
        def preproc(row):
            if "_compressed" not in row["data"]:
                return row

            compressed = row["data"]["_compressed"]
            for key, value in compressed.items():
                assert key not in row["data"]
                row["data"][key] = LbchdbFixture.decompress(value, "brotli")

            row["data"]["_compressed"] = list(compressed.keys())

            return row

        return preproc

    def restore_preproc(self):
        def preproc(row):
            row["data"]["data_json"], compressed = LbchdbFixture.decrypt(row["data"]["data_json"], self._restore_keys)
            if compressed:
                row["__compressed_fields"] = ["data_json"]

            return row

        return preproc

    def yasms_sms_history_preproc(self):
        def preproc(row):
            if "encrypted_text" in row["data"]:
                row["data"]["encrypted_text"], _ = LbchdbFixture.decrypt(
                    row["data"]["encrypted_text"],
                    self._yasms_private_keys,
                )
            return row

        return preproc
