import os
import time

from concurrent.futures import TimeoutError as FutureTimeoutError

from kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api import PQStreamingAPI, ConsumerMessageType
from kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api import ProducerConfigurator, ConsumerConfigurator


class TopicType:
    AUTH = 'auth'
    BB_AUTH = 'bb_auth'
    MAIL_USER_JOURNAL = 'mail_user_journal'
    PUSH = 'push'
    PUSH_SUBSCRIPTION = 'push_subscription'
    SENDR = 'sendr'
    RESTORE = 'restore'
    YASMS_PRIVATE = 'yasms_private'


ALL_TOPICS = [getattr(TopicType, topic) for topic in dir(TopicType) if not topic.startswith('__')]
assert sorted(ALL_TOPICS) == sorted(os.getenv('LOGBROKER_CREATE_TOPICS').split(','))


def port():
    return int(os.getenv('LOGBROKER_PORT'))


def skip_data_for(client_id, topics=ALL_TOPICS):
    lb = TestLogbroker()
    lb.skip_data_for(client_id, topics)
    lb.stop()


class TestLogbroker:
    def __init__(self):
        self._timeout = 10

        self._api = PQStreamingAPI('localhost', port())
        assert self._api and self._api.start().result(timeout=self._timeout)

        self._writers = {}
        self._seq_no = 0

    def stop(self):
        for writer in self._writers.values():
            writer.stop()
        self._api.stop()

    def __get_writer(self, topic):
        if topic not in self._writers:
            configurator = ProducerConfigurator(topic, "test_writer")
            writer = self._api.create_producer(configurator)
            assert writer.start().result(timeout=self._timeout).HasField('init')
            self._writers[topic] = writer
        return self._writers[topic]

    def write(self, topic, data):
        if not data.endswith('\n'):
            data += '\n'

        self._seq_no += 1
        assert self.__get_writer(topic).write(self._seq_no, data).result(timeout=self._timeout).HasField("ack")

    def skip_data_for(self, client_id, topics=ALL_TOPICS):
        readers = []
        for topic in topics:
            configurator = ConsumerConfigurator(topic, client_id)
            reader = self._api.create_consumer(configurator)
            assert reader.start().result(timeout=self._timeout).HasField("init")
            readers.append(reader)

        time.sleep(1)

        for reader in readers:
            cookies = []
            while True:
                try:
                    event = reader.next_event().result(timeout=0)
                    if event.type == ConsumerMessageType.MSG_COMMIT:
                        continue
                    assert event.type == ConsumerMessageType.MSG_DATA
                    cookies.append(event.message.data.cookie)
                except FutureTimeoutError:
                    break
            reader.commit(cookies)
            reader.stop()
