#include "cleaner.h"

namespace NPassport::NLbcPharma {
    TCleaner::TCleaner(std::shared_ptr<NDbPool::TDbPool> db, TCleanerSettings&& settings)
        : Db_(std::move(db))
        , Settings_(settings)
    {
        Task_ = std::make_unique<TRandomSleepTask>(
            [this]() { TCleaner::Run(); },
            Settings_.MaxCleanInterval,
            "cleaner");
    }

    void TCleaner::Run() {
        time_t cutoffTS = NConverter::InvertTs((TInstant::Now() - Settings_.DataTtl).Seconds());
        size_t ops;
        do {
            ops = DeleteBatch("usages", "ts", cutoffTS);
            DeletedRowsUsages_ += ops;
        } while (ops == Settings_.BatchSize);

        do {
            ops = DeleteBatch("bindings", "add_ts", cutoffTS);
            DeletedRowsBindings_ += ops;
        } while (ops == Settings_.BatchSize);
    }

    void TCleaner::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(DeletedRowsUsages_);
        builder.Add(DeletedRowsBindings_);
    }

    size_t TCleaner::DeleteBatch(const TStringBuf table, const TStringBuf column, time_t cutoff) const {
        TString request = NUtils::CreateStr(
            "DELETE FROM ", table, " WHERE ", column, ">", cutoff, " LIMIT ", Settings_.BatchSize);

        size_t affectedRows = 0;

        TInstant start = TInstant::Now();
        try {
            NDbPool::TBlockingHandle handle(*Db_);
            affectedRows = handle.Query(request)->AffectedRows();
        } catch (NDbPool::TException& e) {
            TLog::Warning() << "Cleanup query exception: " << e.what();
            return 0;
        }
        TInstant end = TInstant::Now();

        if (affectedRows > 0) {
            TLog::Debug() << "deleted " << affectedRows << " rows from " << table
                          << " in " << HumanReadable(end - start);
        }
        return affectedRows;
    }
}
