#include "lbcpharma.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/logbroker/processing/reader_pool.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/string/subst.h>
#include <util/system/env.h>

namespace NPassport::NLbcPharma {
    TLbcPharma::TLbcPharma()
        : DbCtx_(NDbPool::TDbPoolCtx::Create())
    {
    }

    TLbcPharma::~TLbcPharma() {
        Stop();
        TLog::Debug() << "LbcPharma was destroyed";
    }

    void TLbcPharma::Init(const NJson::TConfig& config) {
        const TString path = "/component";

        config.InitCommonLog(path + "/logger");
        DbCtx_->InitLogger(config.As<TString>(path + "/dbpool_log"));

        SetEnv("TZ", "Europe/Moscow");
        tzset();

        InitTvm(config, path + "/tvm");
        TLogTypes logTypes = InitLogbroker(config, path + "/logbroker");
        InitDb(config, path + "/db");
        InitDataPusher(config, path + "/data_pusher");
        InitProcessor(config, path + "/processor", std::move(logTypes));
        InitCleaner(config, path + "/cleaner");

        Start();
    }

    void TLbcPharma::HandleRequest(NCommon::TRequest& req) {
        TStringBuf path = req.GetPath();
        path.ChopSuffix("/");

        if (path == "/healthcheck") {
            HandleHealthCheck(req);
            return;
        }

        TLog::Warning() << "Unknown path: " << path;
        req.SetStatus(HTTP_NOT_FOUND);
        req.Write(TStringBuilder() << "Unknown path: " << path);
    }

    void TLbcPharma::AddUnistat(NUnistat::TBuilder& builder) {
        DbCtx_->AddUnistat(builder);
        DataPusher_->AddUnistat(builder);
        Processor_->AddUnistat(builder);
        Lb_->AddUnistat(builder);
        UncommittedMemoryDispatcher_->AddUnistat(builder);
        if (Cleaner_) {
            Cleaner_->AddUnistat(builder);
        }
    }

    void TLbcPharma::InitTvm(const NJson::TConfig& config, const TString& path) {
        const TString secret = config.As<TString>(path + "/self_secret");

        NTvmAuth::NTvmApi::TClientSettings::TDstMap dsts = {
            {NLb::TReaderSettings{}.TvmDstAlias, config.As<ui32>(path + "/dsts/logbroker")},
        };

        NTvmAuth::NTvmApi::TClientSettings setts;
        setts.SetSelfTvmId(config.As<ui32>(path + "/self_tvm_id"));
        setts.EnableServiceTicketsFetchOptions(secret, std::move(dsts));
        setts.SetDiskCacheDir(config.As<TString>(path + "/disk_cache"));

        Tvm_ = std::make_shared<NTvmAuth::TTvmClient>(setts, NTvmLogger::TLogger::Create());

        TLog::Info() << "Tvm inited";
    }

    TLbcPharma::TLogTypes TLbcPharma::InitLogbroker(const NJson::TConfig& config, const TString& path) {
        Lb_ = std::make_shared<TLbReaderPool>(
            NLb::TReaderPoolSettings{
                .QueueSizeLimit = config.As<ui32>(path + "/queue_size"),
                .Workers = config.As<ui32>(path + "/queue_workers"),
                .JoinMessages = config.As<ui32>(path + "/join_messages", 1),
            });

        NLb::TReaderSettings setts;
        setts.LogLevel = config.As<ui32>(path + "/log_level");
        setts.MaxMemoryUsage = 1024 * 1024 * config.As<size_t>(path + "/memory_usage_per_server__mb");
        setts.ReadMaxBytes = config.As<size_t>(path + "/read_max_bytes");
        setts.ReadMaxCounts = config.As<size_t>(path + "/read_max_count", setts.ReadMaxCounts);
        setts.InflightReads = config.As<size_t>(path + "/inflight_reads", setts.InflightReads);
        setts.MaxUncommittedSize = config.As<size_t>(path + "/max_uncommitted_size", setts.MaxUncommittedSize);
        setts.ClientId = config.As<TString>(path + "/client_id");

        setts.TvmClient = Tvm_;

        UncommittedMemoryDispatcher_ = setts.UncommittedMemory;

        TLogTypes logTypes;
        for (const TString& pathPrefix : config.SubKeys(path + "/logs")) {
            const TString type = NJson::TConfig::GetKeyFromPath(pathPrefix);

            // throw if type is unknown
            TLogTypeTraits::FromString(type);

            logTypes.push_back(type);
            setts.Topics.push_back(config.As<TString>(pathPrefix));
        }

        for (const TString& serverName : config.As<std::vector<TString>>(path + "/servers")) {
            setts.ServerName = serverName;
            setts.UnistatSignalId = serverName;

            Lb_->AddReader(setts);
        }

        Y_ENSURE(Lb_->GetReadersCount(), "There is no one server in config");

        TLog::Info() << "LB inited";
        return logTypes;
    }

    void TLbcPharma::InitDb(const NJson::TConfig& config, const TString& path) {
        Db_ = DbCtx_->CreateDb(config, path);
        Db_->TryPing();

        TLog::Info() << "DbPool inited";
    }

    void TLbcPharma::InitDataPusher(const NJson::TConfig& config, const TString& path) {
        DataPusher_ = std::make_unique<TDataPusher>(Lb_);

        TLog::Debug() << "TDataPusher created";

        const size_t workers = config.As<ui32>(path + "/workers", 1);
        DataPusher_->CreateWorkers<TDataPushWorker>(
            workers,
            TPusherSettings{
                .Db = Db_.get(),
            });
        TLog::Info() << "DataPusher use " << workers << " workers";
    }

    void TLbcPharma::InitProcessor(const NJson::TConfig& config, const TString& path, TLogTypes&& logTypes) {
        const TString badLinesLog = config.As<TString>(path + "/bad_lines_logfile_pattern");
        const size_t batchSize = config.As<size_t>(path + "/batch_size");

        NLb::TBadLineLoggers<TLogTypeTraits> loggers;
        loggers.Init(badLinesLog, logTypes);

        Processor_ = std::make_unique<TProcessor>(
            TProcessorSettings{
                .BatchSize = batchSize,
            },
            *Db_,
            std::move(loggers));
    }

    void TLbcPharma::InitCleaner(const NJson::TConfig& config, const TString& path) {
        if (!config.As<bool>(path + "/enabled")) {
            TLog::Info() << "Cleaner disabled.";
            return;
        }

        const ui64 max_clean_interval_msecs = config.As<ui64>(path + "/max_clean_interval__msec");
        const ui64 data_ttl_days = config.As<ui64>(path + "/data_ttl__days");
        const ui64 batch_size = config.As<ui64>(path + "/batch_size");

        Cleaner_ = std::make_unique<TCleaner>(
            Db_,
            TCleanerSettings{
                .MaxCleanInterval = TDuration::MilliSeconds(max_clean_interval_msecs),
                .DataTtl = TDuration::Days(data_ttl_days),
                .BatchSize = batch_size,
            });

        TLog::Info() << "Log cleaner inited";
    }

    static const TString CONTENT_TYPE = "Content-type";
    static const TString CONTENT_TYPE_TEXT = "text/plain";

    void TLbcPharma::HandleHealthCheck(NCommon::TRequest& req) {
        req.Write(Tvm_->GetStatus().CreateJugglerMessage());
        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_TEXT);
        req.SetStatus(HTTP_OK);
    }

    void TLbcPharma::Start() {
        Lb_->Start(
            "lbcph_",
            [this](const NLb::TDataSet<>& data) {
                return Processor_->Proc(data);
            });
    }

    void TLbcPharma::Stop() {
        Lb_.reset(); // stop readers which push this data pipeline
        Tvm_.reset();

        DataPusher_.reset(); // stop parsers - last users of db
    }
}
