#include "processor.h"

#include "converters/bindings.h"
#include "converters/yasms_private.h"
#include "parsers/bindings.h"
#include "parsers/yasms_private.h"

#include <passport/infra/libs/cpp/dbpool/handle.h>

#include <ydb/library/persqueue/topic_parser/topic_parser.h>

namespace NPassport::NLbcPharma {
    static const std::map<ELogType, TString> LOG_TYPES = {
        {ELogType::Bindings, "bindings"},
        {ELogType::YasmsPrivate, "yasms_private"},
    };

    const TString& TLogTypeTraits::ToString(ELogType type) {
        auto it = LOG_TYPES.find(type);
        Y_VERIFY(it != LOG_TYPES.end());
        return it->second;
    }

    ELogType TLogTypeTraits::FromString(const TStringBuf type) {
        for (const auto& [e, s] : LOG_TYPES) {
            if (type == s) {
                return e;
            }
        }

        ythrow yexception() << "Unknown log type: " << type;
    }

    TProcessor::TProcessor(const TProcessorSettings& settings,
                           NDbPool::TDbPool& db,
                           NLb::TBadLineLoggers<TLogTypeTraits> badLinesLoggers)
        : TBase(std::move(badLinesLoggers))
        , Settings_(settings)
        , UnistatBindingsReadLag_(
              "read_lag.bindings",
              NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::Minutes(10)))
        , UnistatYasmsPrivateReadLag_(
              "read_lag.yasms_private",
              NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::Minutes(10)))
        , Db_(db)
    {
    }

    TProcessor::~TProcessor() = default;

    void TProcessor::AddUnistat(NUnistat::TBuilder& builder) const {
        TBase::AddUnistat(builder);
        UnistatBindingsReadLag_.AddUnistat(builder);
        UnistatYasmsPrivateReadLag_.AddUnistat(builder);
    }

    TQueries TProcessor::Proc(const NLb::TDataSet<>& data) {
        TQueries res(Settings_.BatchSize);

        for (const NLb::TDataWithType<>& d : data) {
            for (const NLb::TTopicData& td : d.Data.Messages) {
                res.Merge(ChooseProc(td));
            }
        }

        return res;
    }

    TQueries TProcessor::ChooseProc(const NLb::TTopicData& td) {
        TString topic = NPersQueue::GetTopicPath(td.Topic);

        switch (TLogTypeTraits::FromString(TStringBuf(topic).RNextTok("/"))) {
            case ELogType::Bindings:
                return ProcBindings(td);
            case ELogType::YasmsPrivate:
                return ProcYasmsPrivate(td);
            case ELogType::COUNT:
                Y_FAIL("Internal error");
        }
    }

    TQueries TProcessor::ProcBindings(const NLb::TTopicData& data) {
        NDbPool::TBlockingHandle sql(Db_);

        NParser::TBindingsParser parser(&BadLinesLoggers_.GetLogger(ELogType::Bindings));
        NConverter::TBindingsConverter conv(sql);

        NParser::TBindingsParser::TResult res = parser.Parse(data);
        CountUnistatSignals(ELogType::Bindings, parser);

        conv.Reserve(res.size());

        for (NParser::TBindingsRow& r : res) {
            UnistatBindingsReadLag_.Insert(TInstant::Now() - TInstant::Seconds(r.Ts));
            conv.Process(std::move(r));
        }

        return conv.Finish();
    }

    TQueries TProcessor::ProcYasmsPrivate(const NLb::TTopicData& data) {
        NDbPool::TBlockingHandle sql(Db_);

        NParser::TYasmsPrivateParser parser(&BadLinesLoggers_.GetLogger(ELogType::YasmsPrivate));
        NConverter::TYasmsPrivateConverter conv(sql);

        NParser::TYasmsPrivateParser::TResult res = parser.Parse(data);
        CountUnistatSignals(ELogType::YasmsPrivate, parser);

        conv.Reserve(res.size());

        for (NParser::TYasmsPrivateRow& r : res) {
            UnistatYasmsPrivateReadLag_.Insert(TInstant::Now() - TInstant::Seconds(r.Ts));
            conv.Process(std::move(r));
        }

        return conv.Finish();
    }
}
