#include "query.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NLbcPharma {
    void TQueries::Reserve(const TString& tableName, const TQueryBuilderPtr& builder, size_t size) {
        TBatches& batches = MutateBatches(tableName, builder);

        if (BatchSize_) {
            batches.reserve(size / BatchSize_ + 1);
        } else {
            if (batches.empty()) {
                batches.push_back({});
            }
            batches.back().reserve(size);
        }
    }

    void TQueries::Add(const TString& tableName, const TQueryBuilderPtr& builder, TQuery&& query) {
        TBatches& batches = MutateBatches(tableName, builder);

        if (batches.empty() || (BatchSize_ && batches.back().size() >= BatchSize_)) {
            batches.push_back({});
            if (BatchSize_) {
                batches.back().reserve(BatchSize_);
            }
        }

        batches.back().push_back(std::move(query));
        ++Size_;
    }

    void TQueries::Merge(TQueries&& queries) {
        if (BatchSize_ == 0) {
            std::move(queries.Data_.begin(), queries.Data_.end(), Data_.end());
            return;
        }

        for (TPerTable& p : queries.Data_) {
            for (std::vector<TQuery>& b : p.Batches) {
                for (TQuery& q : b) {
                    Add(p.TableName, p.Builder, std::move(q));
                }
            }
        }
    }

    std::vector<TString> TQueries::BuildSql() const {
        std::vector<TString> res;

        size_t batches = 0;
        for (const TPerTable& p : Data_) {
            batches += p.Batches.size();
        }
        res.reserve(batches);

        for (const TPerTable& p : Data_) {
            for (const std::vector<TQuery>& b : p.Batches) {
                if (!b.empty()) {
                    res.push_back(p.Builder->Build(b));
                }
            }
        }

        return res;
    }

    size_t TQueries::size() const {
        return Size_;
    }

    TQueries::TBatches& TQueries::MutateBatches(const TString& tableName, const TQueryBuilderPtr& builder) {
        auto it = std::find_if(Data_.begin(), Data_.end(), [&](const TPerTable& o) {
            return o.TableName == tableName;
        });

        if (it != Data_.end()) {
            return it->Batches;
        }

        Data_.push_back(TPerTable{
            .TableName = tableName,
            .Builder = builder,
        });

        return Data_.back().Batches;
    }

    TString TBaseQueryBuilder::BuildImpl(const std::vector<TQuery>& queries,
                                         TStringBuf header,
                                         TStringBuf footer) {
        Y_ENSURE(!queries.empty());

        TString res = NUtils::CreateStrExt(
            queries.front().SerializedEntity.size() * queries.size() + header.size() + footer.size(),
            header);

        for (const TQuery& q : queries) {
            NUtils::Append(res, q.SerializedEntity, ",");
        }

        res.pop_back();
        res.append(footer);

        return res;
    }

}
