#include <passport/infra/daemons/lbcpharma/src/parsers/bindings.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLbcPharma;
using namespace NPassport::NLbcPharma::NParser;

Y_UNIT_TEST_SUITE(BindingsParser) {
    class TTestBindingsParser: public NParser::TBindingsParser {
    public:
        using TBindingsParser::ParseRow;
    };

    Y_UNIT_TEST(parse) {
        TTestBindingsParser parser;

        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow(""),
                                       yexception,
                                       "line should start with 'tskv'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=something"),
                                       yexception,
                                       "format 'something' is unknown, supported: pharma");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma"),
                                       yexception,
                                       "no field 'action'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=kek"),
                                       yexception,
                                       "action 'kek' is unknown, supported: confirmed");

        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed"),
                                       yexception,
                                       "no field 'phonenumber'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "phonenumber=asd"),
                                       yexception,
                                       "invalid number in 'phonenumber': 'asd'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "phonenumber=123"),
                                       yexception,
                                       "no field 'ts'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "phonenumber=123\t"
                                                       "ts=asd"),
                                       yexception,
                                       "invalid number in 'ts': 'asd'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "phonenumber=123\t"
                                                       "ts=456"),
                                       yexception,
                                       "no field 'uid'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "phonenumber=123\t"
                                                       "uid=asd\t"
                                                       "ts=654"),
                                       yexception,
                                       "invalid number in 'uid': 'asd'");

        UNIT_ASSERT_NO_EXCEPTION(parser.ParseRow("tskv\t"
                                                 "tskv_format=pharma\t"
                                                 "action=confirmed\t"
                                                 "uid=70500\t"
                                                 "phonenumber=123\t"
                                                 "ts=456"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(parser.ParseRow("tskv\t"
                                                       "tskv_format=pharma\t"
                                                       "action=confirmed\t"
                                                       "uid=70500\t"
                                                       "phonenumber=123\t"
                                                       "ts=456\t"
                                                       "user_ip=asdas"),
                                       yexception,
                                       "invalid ip address in 'user_ip': 'asdas'");

        UNIT_ASSERT_NO_EXCEPTION(parser.ParseRow("tskv\t"
                                                 "tskv_format=pharma\t"
                                                 "action=confirmed\t"
                                                 "uid=70500\t"
                                                 "phonenumber=789\t"
                                                 "ts=654\t"
                                                 "user_ip=127.0.0.1\t"
                                                 "user_agent=asdas"));

        TBindingsParser::TResult result = parser.Parse({});

        TBindingsRow row1{TBaseRow{
            .Phonenum = 123,
            .Ts = 456,
        }};
        row1.Uid = 70500;

        TBindingsRow row2{TBaseRow{
            .Phonenum = 789,
            .UserIp = NUtils::TIpAddr("127.0.0.1"),
            .UserAgent = "asdas",
            .Ts = 654,
        }};
        row2.Uid = 70500;

        UNIT_ASSERT_VALUES_EQUAL(
            result,
            TBindingsParser::TResult({row1, row2}));

        UNIT_ASSERT_NO_EXCEPTION(parser.ParseRow("tskv	action=confirmed	phonenumber=70000920925	ts=1606914564	tskv_format=pharma	uid=4043234521	user_agent=Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/83.0.4103.61 Safari/537.36	user_ip=2a02:6b8:c13:3408:0:1400:93d3:0"));
    }
}

template <>
void Out<TBindingsRow>(IOutputStream& o, const TBindingsRow& value) {
    o << "num=" << value.Phonenum << ";";
    o << "ip=" << (value.UserIp ? value.UserIp->ToString() : TString()) << ";";
    o << "agent=" << value.UserAgent << ";";
    o << "ts=" << value.Ts << ";";
    o << "uid=" << value.Uid << ";";
}

template <>
void Out<TBindingsParser::TResult>(IOutputStream& o, const TBindingsParser::TResult& value) {
    for (const auto& r : value) {
        o << r << Endl;
    }
}
