#include "runtime_context.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/regex/glob/glob_iterator.h>

#include <util/folder/filelist.h>
#include <util/folder/path.h>
#include <util/generic/yexception.h>
#include <util/system/hostname.h>

#include <unordered_map>

namespace NPassport::NLogstoreAgent {
    TRuntimeContext::TRuntimeContext(const NJson::TConfig& config, const TString& jpoint)
        : DbPoolLogFile_(config.As<TString>(jpoint + "/dbpool_log"))
    {
        InitDefaultSettings(config, jpoint);
        InitDbSettings(config, jpoint);
        InitInstances(config, jpoint);
        InitTvm(config, jpoint);
    }

    TRuntimeContext::~TRuntimeContext() = default;

    void TRuntimeContext::InitLogger(const TString& filename) {
        TLog::Init(std::make_unique<NUtils::TFileLogger>(filename, "DEBUG", true, "_DEFAULT_"));
    }

    TContext TRuntimeContext::PrepareContext() const {
        std::unordered_map<TString, size_t> sizes;
        TContext::TConfigsByFilenameMap configs;

        std::shared_ptr<NDbPool::TDbPoolCtx> ctx = NDbPool::TDbPoolCtx::Create();
        ctx->InitLogger(DbPoolLogFile_);

        for (const TInstanceConfig& ins : Instances_) {
            TGlobPaths g(ins.Filename);
            for (TString path : g) {
                TLogStreamConfig perFile{
                    .Ins = ins,
                    .CacheDir = CacheDir_ / ins.DestinationHost / TFsPath(path).RelativePath("/"),
                    .SourceHost = Hostname_,
                    .StreamId = NUtils::CreateStr(ins.DestinationHost, ":", path),
                };
                perFile.Ins.Filename = path;

                auto it = configs.try_emplace(path, TContext::TConfigByHostnameMap{}).first;
                bool inserted = it->second.try_emplace(ins.DestinationHost, std::move(perFile)).second;
                if (!inserted) {
                    TLog::Warning() << "Duplicate file '" << path << "' for host: " << ins.DestinationHost;
                    continue;
                }

                sizes[ins.DestinationHost] += ins.Common.ParallelPushesPerLog;
            }
        }

        std::map<TString, std::shared_ptr<NDbPool::TDbPool>> pools;
        for (const auto& [hostname, size] : sizes) {
            pools.emplace(hostname, CreatePool(ctx, hostname, size));
        }

        for (auto& [filename, streams] : configs) {
            for (auto& [hostname, stream] : streams) {
                stream.Pool = pools[stream.Ins.DestinationHost];
            }
        }

        return TContext{
            .Ctx = std::move(ctx),
            .Configs = std::move(configs),
        };
    }

    bool TRuntimeContext::CheckContext(std::shared_ptr<TContext> context) const {
        if (!context) {
            return false;
        }

        for (const TInstanceConfig& ins : Instances_) {
            TGlobPaths g(ins.Filename);
            for (TString path : g) {
                if (!context->Configs.contains(path)) {
                    return false;
                }
            }
        }
        return true;
    }

    TDuration TRuntimeContext::GetFilesRescanPeriod() const {
        return FilesRescanPeriod_;
    }

    static const THashMap<TString, TCommonConfig::ECompressionCodec> CODECS = {
        {"plaintext", TCommonConfig::ECompressionCodec::PlainText},
        {"gzip", TCommonConfig::ECompressionCodec::GZip},
        {"zstd", TCommonConfig::ECompressionCodec::ZStd5},
        {"zstd0", TCommonConfig::ECompressionCodec::ZStd0},
        {"zstd3", TCommonConfig::ECompressionCodec::ZStd3},
        {"zstd5", TCommonConfig::ECompressionCodec::ZStd5},
    };

    static TCommonConfig::ECompressionCodec CastCodec(const TString& codec) {
        auto it = CODECS.find(codec);
        Y_ENSURE(it != CODECS.end(),
                 "codec is unknown: '" << codec << "'");
        return it->second;
    }

    static TString CastCodec(TCommonConfig::ECompressionCodec codec) {
        for (const auto& [str, e] : CODECS) {
            if (e == codec) {
                return str;
            }
        }

        Y_FAIL("Unreachable case: codec is unknown: %s", TString(TStringBuilder() << codec).c_str());
    }

    static const THashMap<TString, TCommonConfig::ETimeAggregation> AGGREGATIONS = {
        {"day", TCommonConfig::ETimeAggregation::Day},
        {"hour", TCommonConfig::ETimeAggregation::Hour},
    };

    static TCommonConfig::ETimeAggregation CastAggregationType(const TString& type) {
        auto it = AGGREGATIONS.find(type);
        Y_ENSURE(it != AGGREGATIONS.end(),
                 "aggregation type is unknown: '" << type << "'");
        return it->second;
    }

    static TString CastAggregationType(TCommonConfig::ETimeAggregation type) {
        for (const auto& [str, e] : AGGREGATIONS) {
            if (e == type) {
                return str;
            }
        }

        Y_FAIL("Unreachable case: aggregation type is unknown: %s", TString(TStringBuilder() << type).c_str());
    }

    void TRuntimeContext::InitDefaultSettings(const NJson::TConfig& config, const TString& jpoint) {
        DefaultSettings_ = TCommonConfig{
            .Env = config.As<TString>(jpoint + "/defaults/env"),
            .ChunkSize = std::min(config.As<size_t>(jpoint + "/defaults/chunk_size__KB"), 8096ul),
            .ChunkCollectTimeout = TDuration::MilliSeconds(
                config.As<ui64>(jpoint + "/defaults/chunk_collect_timeout__msec")),
            .CompressionCodec = CastCodec(config.As<TString>(jpoint + "/defaults/compression_codec")),
            .ParallelPushesPerLog = config.As<size_t>(jpoint + "/defaults/parallel_pushes_per_log"),
            .Aggregation = CastAggregationType(config.As<TString>(jpoint + "/defaults/rotation_type")),
            .RotationTimeout = TDuration::Seconds(config.As<ui64>(jpoint + "/defaults/rotation_timeout__secs")),
            .ForceRotationTimeout = TDuration::Seconds(config.As<ui64>(jpoint + "/defaults/rotation_timeout_hard__secs")),
            .MaxHistorySize = config.As<ui64>(jpoint + "/max_history_size", 1440),
            .HistoryUpdatePeriod = TDuration::Minutes(config.As<ui64>(jpoint + "/history_update_period__mins", 1)),
            .ReadLagWarningThreshold = TDuration::Minutes(config.As<ui64>(jpoint + "/read_lag_warning_threshold__mins", 120)),
            .ReadLagCriticalThreshold = TDuration::Minutes(config.As<ui64>(jpoint + "/read_lag_critical_threshold__mins", 240)),
        };

        OverrideDestinationHost_ = config.As<TString>(jpoint + "/override_api_host", "");
        Hostname_ = config.As<TString>(jpoint + "/hostname", FQDNHostName());
        CacheDir_ = config.As<TString>(jpoint + "/cache_dir");
        FilesRescanPeriod_ = TDuration::Seconds(config.As<ui64>(jpoint + "/files_rescan_period__sec", 60));
    }

    void TRuntimeContext::InitDbSettings(const NJson::TConfig& config, const TString& jpoint) {
        DbSettings_ = TDbSettings{
            .GetTimeout = TDuration::MilliSeconds(config.As<ui64>(jpoint + "/pools/get_timeout")),
            .ConnectTimeout = TDuration::MilliSeconds(config.As<ui64>(jpoint + "/pools/connect_timeout")),
            .QueryTimeout = TDuration::MilliSeconds(config.As<ui64>(jpoint + "/pools/query_timeout")),
            .UseTls = config.As<bool>(jpoint + "/pools/use_tls"),
            .Port = (ui16)config.As<ui32>(jpoint + "/pools/port"),
        };
    }

    void TRuntimeContext::InitInstances(const NJson::TConfig& config, const TString& jpoint) {
        const std::vector<TString> skipEnvironments = config.As<std::vector<TString>>(jpoint + "/skip_envs");
        const std::vector<TString> filenames =
            GetFilenamesWithInstances(config.As<TString>(jpoint + "/instances"));

        for (const TString& file : filenames) {
            for (TInstanceConfig& instanceConfig : ReadInstances(file)) {
                auto it = std::find_if(skipEnvironments.begin(),
                                       skipEnvironments.end(),
                                       [&env = instanceConfig.Common.Env](const TStringBuf key) {
                                           return env.Contains(key);
                                       });

                if (it != skipEnvironments.end()) {
                    TLog::Info() << "Skipping env '" << instanceConfig.Common.Env << "': " << instanceConfig.Filename;
                    continue;
                }
                Instances_.push_back(std::move(instanceConfig));
            }
        }
    }

    void TRuntimeContext::InitTvm(const NJson::TConfig& config, const TString& jpoint) {
        NTvmAuth::NTvmApi::TClientSettings settings;

        settings.SetSelfTvmId(config.As<ui32>(jpoint + "/tvm/self_tvm_id"));

        NTvmAuth::NTvmApi::TClientSettings::TDstMap dsts;

        for (const TString& host : config.SubKeys(jpoint + "/tvm/dsts")) {
            dsts.emplace(NJson::TConfig::GetKeyFromPath(host),
                         config.As<ui32>(host));
        }

        for (const TInstanceConfig& ins : Instances_) {
            Y_ENSURE(dsts.contains(ins.DestinationHost),
                     "There is no service-ticket for host: '" << ins.DestinationHost << "'");
        }

        settings.EnableServiceTicketsFetchOptions(
            NUtils::ReadFile(config.As<TString>(jpoint + "/tvm/self_secret_file")),
            std::move(dsts));
        settings.SetDiskCacheDir(config.As<TString>(jpoint + "/tvm/disk_cache"));

        settings.SetTvmHostPort(
            config.As<TString>(jpoint + "/tvm/tvm_host", settings.TvmHost),
            config.As<ui32>(jpoint + "/tvm/tvm_port", settings.TvmPort));

        Tvm_ = std::make_shared<NTvmAuth::TTvmClient>(settings, NTvmLogger::TLogger::Create());

        TLog::Info() << "TVM inited";
    }

    std::vector<TString> TRuntimeContext::GetFilenamesWithInstances(const TString& dir) {
        TFileList list;
        list.Fill(dir);

        std::vector<TString> res;

        TString name;
        while ((name = list.Next()) != nullptr) {
            res.push_back(JoinFsPaths(dir, name));
        }

        return res;
    }

    std::vector<TInstanceConfig> TRuntimeContext::ReadInstances(const TString& filepath) const {
        TLog::Debug() << "Reading config from " << filepath;
        const NJson::TConfig config = NJson::TConfig::ReadFromFile(filepath);

        std::vector<TString> filenames;
        try {
            filenames.emplace_back(config.As<TString>("/file"));
        } catch (const NJson::TConfig::TBadValueException& e) {
            filenames = config.As<std::vector<TString>>("/file");
        }

        std::vector<TString> hosts;
        try {
            hosts.emplace_back(config.As<TString>("/host"));
        } catch (const NJson::TConfig::TBadValueException& e) {
            hosts = config.As<std::vector<TString>>("/host");
        }

        std::vector<TInstanceConfig> instances;
        for (const TString& filename : filenames) {
            for (const TString& host : hosts) {
                instances.emplace_back(TInstanceConfig{
                    .Filename = filename,
                    .DestinationHost = OverrideDestinationHost_ ? OverrideDestinationHost_ : host,
                    .Common = TCommonConfig{
                        .Env = config.As<TString>("/env", DefaultSettings_.Env),
                        .ChunkSize = std::min(config.As<size_t>("/chunk_size__KB", DefaultSettings_.ChunkSize), 8096ul),
                        .ChunkCollectTimeout = TDuration::MilliSeconds(
                            config.As<ui64>("/chunk_collect_timeout__msec", DefaultSettings_.ChunkCollectTimeout.MilliSeconds())),
                        .CompressionCodec = CastCodec(config.As<TString>(
                            "/compression_codec",
                            CastCodec(DefaultSettings_.CompressionCodec))),
                        .ParallelPushesPerLog = config.As<size_t>("/parallel_pushes_per_log", DefaultSettings_.ParallelPushesPerLog),
                        .Aggregation = CastAggregationType(config.As<TString>(
                            "/rotation_type",
                            CastAggregationType(DefaultSettings_.Aggregation))),
                        .RotationTimeout = TDuration::Seconds(config.As<ui64>("/rotation_timeout__secs", DefaultSettings_.RotationTimeout.Seconds())),
                        .ForceRotationTimeout = TDuration::Seconds(config.As<ui64>("/rotation_timeout_hard__secs", DefaultSettings_.RotationTimeout.Seconds())),
                        .MaxHistorySize = DefaultSettings_.MaxHistorySize,
                        .HistoryUpdatePeriod = DefaultSettings_.HistoryUpdatePeriod,
                        .ReadLagWarningThreshold = DefaultSettings_.ReadLagWarningThreshold,
                        .ReadLagCriticalThreshold = DefaultSettings_.ReadLagCriticalThreshold,
                    },
                });
            }
        }
        return instances;
    }

    std::shared_ptr<NDbPool::TDbPool> TRuntimeContext::CreatePool(
        std::shared_ptr<NDbPool::TDbPoolCtx> ctx,
        const TString& destinationHost,
        size_t size) const {
        NDbPool::TDbPoolSettings s{
            .Dsn = NDbPool::TDestination::CreateHttp(destinationHost),
            .Hosts = {
                NDbPool::TDbHost{
                    .Host = NUtils::CreateStr(
                        DbSettings_.UseTls ? "https" : "http",
                        "://",
                        destinationHost),
                    .Port = DbSettings_.Port,
                },
            },
            .Size = size,
            .GetTimeout = DbSettings_.GetTimeout,
            .ConnectionTimeout = DbSettings_.ConnectTimeout,
            .QueryTimeout = DbSettings_.QueryTimeout,
            .DefaultQueryOpts = std::make_shared<NDbPool::TQueryOpts>(),
        };

        s.DefaultQueryOpts->push_back(NTvmDbPool::CreateServiceTicketOpt(
            Tvm_,
            destinationHost));

        return ctx->CreateDb(s);
    }
}
