#include "tailer.h"

#include <passport/infra/libs/cpp/juggler/status.h>

#include <library/cpp/streams/zstd/zstd.h>

#include <util/stream/zlib.h>

namespace NPassport::NLogstoreAgent {
    TTailer::TTailer(TTailerSettings&& settings)
        : Callback_(std::move(settings.Callback))
        , TailerId_(std::move(settings.TailerId))
        , BufferLimit_(settings.ChunkSize << 10)
        , ReadLagWarningThreshold_(settings.ReadLagWarningThreshold)
        , ReadLagCriticalThreshold_(settings.ReadLagCriticalThreshold)
    {
        Y_ENSURE(Callback_, "Callback must be initialized");
        InitReader(settings);
    }

    NJuggler::TStatus TTailer::GetJugglerStatus() const {
        TDuration readLag = Reader_->GetLag();
        if (readLag > ReadLagCriticalThreshold_) {
            return NJuggler::TStatus(NJuggler::ECode::Critical, "Read lag exceeded ", readLag.Minutes(), " min. for ", TailerId_);
        }
        if (readLag > ReadLagWarningThreshold_) {
            return NJuggler::TStatus(NJuggler::ECode::Warning, "Read lag exceeded ", readLag.Minutes(), " min. for ", TailerId_);
        }

        return {};
    }

    ETaskStatus TTailer::Run() {
        if (!NextChunk_) {
            std::optional<TChunk> chunk = Reader_->Read(BufferLimit_);
            if (!chunk) {
                return ETaskStatus::Idle;
            }
            NextChunk_ = std::make_shared<TChunk>(std::move(*chunk));
        }

        if ((*Callback_)(NextChunk_)) {
            NextChunk_.reset();
        }
        return ETaskStatus::Success;
    }

    void TTailer::InitReader(const TTailerSettings& settings) {
        Reader_ = std::make_unique<NTail::TReader>(NTail::TReaderSettings{
            .Path = settings.Filename,

            .MissingPolicy = NTail::EMissingPolicy::Ok,
            .BeginningPolicy = NTail::EBeginningPolicy::ContinueFromByte,
            .EmptyBufferPolicy = NTail::EEmptyBufferPolicy::OnINodeChange,

            .Offset = settings.Offset,
            .INode = settings.INode,
            .RotationTimeout = settings.RotationTimeout,
            .ForceRotationTimeout = settings.ForceRotationTimeout,

            .History = NTail::THistorySettings{
                .CachePath = settings.CacheDir / "history",
                .MaxHistorySize = settings.MaxHistorySize,
                .UpdatePeriod = settings.HistoryUpdatePeriod,
            },
        });
    }
}
