#include "log_stream.h"

#include "input/tailer.h"
#include "output/logstoreapi_stream.h"
#include "output/scheduler.h"

#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/unistat/name_factory.h>

#include <util/system/file.h>

namespace NPassport::NLogstoreAgent {
    TLogStream::TLogStream(const TLogStreamConfig& config, const NUnistat::TNameFactory::TTags& unistatTags)
        : Settings_(config)
        , UnistatTags_(unistatTags)
        , Shutdown_(false)
    {
        LastError_.Set(std::make_shared<TString>());
        Settings_.CacheDir.MkDirs();
        InitUnistat();
        Thread_ = std::thread([this]() { Start(); });
    }

    TLogStream::~TLogStream() {
        Shutdown_ = true;
        DestroyState();
        Thread_.join();
    }

    void TLogStream::Start() {
        while (true) {
            TVector<NThreading::TFuture<void>> futures;
            {
                std::unique_lock lock(ReinitializationLock_);
                InitScheduler();
                InitInput();

                futures = {
                    Scheduler_->ShutdownHook(),
                    Tailer_->ShutdownHook(),
                };
            }

            NThreading::WaitAny(futures).Wait();

            DestroyState();

            if (Shutdown_) {
                return;
            }

            TString err = NUtils::CreateStr("Stream had to be reinitialized for ", Settings_.StreamId);
            TLog::Error() << err;
            LastError_.Set(std::make_shared<TString>(std::move(err)));
            Sleep(TDuration::Seconds(5));
        }
    }

    NJuggler::TStatus TLogStream::GetJugglerStatus() const {
        NJuggler::TStatus status;

        std::shared_ptr<TString> error = LastError_.Get();
        if (!error->empty()) {
            status.Update(NJuggler::ECode::Warning, *error);
        }

        std::shared_lock lock(ReinitializationLock_);

        if (Scheduler_) {
            status.Update(Scheduler_->GetJugglerStatus());
        }
        if (Tailer_) {
            status.Update(Tailer_->GetJugglerStatus());
        }

        return status;
    }

    void TLogStream::AddUnistat(NUnistat::TBuilder& builder) {
        builder.Add(*SignalTotalChunks_);
        builder.Add(*SignalTotalBytes_);

        std::shared_lock lock(ReinitializationLock_);

        if (Scheduler_) {
            Scheduler_->AddUnistat(builder);
        }
    }

    void TLogStream::InitInput() {
        TLog::Debug() << "Stream " << Settings_.StreamId << ": tailer init started";

        const auto& common = Settings_.Ins.Common;

        IStateHolder::TState state = Scheduler_->GetState();

        Tailer_ = TTailerTask::CreateUnique(
            TTaskSettings{
                .BackoffSettings = TBackoffSettings{
                    .MinTimeout = TDuration::MilliSeconds(500),
                },
                .Name = NUtils::CreateStr("inp-", Settings_.StreamId),
            },
            TTailerSettings{
                .Filename = Settings_.Ins.Filename,
                .TailerId = Settings_.StreamId,

                .Callback = std::make_shared<TTailerCallback>([this](TChunkPtr chunk) { return ProcessChunk(std::move(chunk)); }),
                .Offset = state.Offset,
                .INode = state.Inode,

                .ChunkSize = common.ChunkSize,
                .RotationTimeout = common.RotationTimeout,
                .ForceRotationTimeout = common.ForceRotationTimeout,

                .CacheDir = Settings_.CacheDir,
                .MaxHistorySize = common.MaxHistorySize,
                .HistoryUpdatePeriod = common.HistoryUpdatePeriod,
                .ReadLagWarningThreshold = common.ReadLagWarningThreshold,
                .ReadLagCriticalThreshold = common.ReadLagCriticalThreshold,
            });

        TLog::Debug() << "Stream " << Settings_.StreamId << ": tailer init completed";
    }

    void TLogStream::InitScheduler() {
        TLog::Debug() << "Stream " << Settings_.StreamId << ": scheduler init started";

        const auto& common = Settings_.Ins.Common;

        auto logstoreApiStream = std::make_unique<TLogstoreApiStream>(
            Settings_.Pool,
            TLogstoreApiStreamSettings{
                .ClientId = Settings_.StreamId,
                .File = Settings_.Ins.Filename.RelativePath("/"),
                .Host = Settings_.SourceHost,
                .Env = common.Env,
                .Codec = common.CompressionCodec,
                .Aggregation = common.Aggregation,
            });

        Scheduler_ = TSchedulerTask::CreateUnique(
            TTaskSettings{
                .BackoffSettings = TBackoffSettings{
                    .Policy = EBackoffPolicy::Exponential,
                    .MinTimeout = TDuration::MilliSeconds(500),
                    .MaxTimeout = TDuration::Seconds(20),
                    .Jitter = TDuration::MilliSeconds(100),
                },
                .Name = NUtils::CreateStr("out-", Settings_.StreamId),
            }, std::move(logstoreApiStream),
            TSchedulerSettings{
                .CacheDir = Settings_.CacheDir,
                .SchedulerId = Settings_.StreamId,
                .BufferSize = 3 * common.ParallelPushesPerLog,
                .MaxQueueSize = common.ParallelPushesPerLog,
            }, UnistatTags_);

        TLog::Debug() << "Stream " << Settings_.StreamId << ": scheduler init completed";
    }

    void TLogStream::DestroyState() {
        TLog::Debug() << "Stream " << Settings_.StreamId << ": destroy started";

        std::unique_lock lock(ReinitializationLock_);

        // Tailer calls Scheduler_ from ProcessChunk() - so it must be stopped first
        if (Tailer_) {
            Tailer_.reset();
            TLog::Debug() << "Stream " << Settings_.StreamId << ": tailer stopped";
        }
        if (Scheduler_) {
            Scheduler_.reset();
            TLog::Debug() << "Stream " << Settings_.StreamId << ": scheduler stopped";
        }
    }

    void TLogStream::InitUnistat() {
        NUnistat::TNameFactory helper("log_stream", UnistatTags_);

        SignalTotalBytes_ = std::make_unique<NUnistat::TSignalDiff<ui64>>(helper.Name("total_bytes"));
        SignalTotalChunks_ = std::make_unique<NUnistat::TSignalDiff<ui64>>(helper.Name("total_chunks"));
    }

    bool TLogStream::ProcessChunk(TChunkPtr chunk) {
        size_t chunkSize = chunk->Data.size();

        if (!Scheduler_->TryPush(std::move(chunk), TDuration::Seconds(5))) {
            return false;
        }
        Scheduler_->RunNow();

        *SignalTotalBytes_ += chunkSize;
        ++(*SignalTotalChunks_);

        return true;
    }
}
