#include "log_stream_manager.h"

#include <passport/infra/daemons/logstoreagent/src/cfg/runtime_context.h>

#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>

namespace NPassport::NLogstoreAgent {
    TLogStreamManager::TLogStreamManager(std::shared_ptr<TRuntimeContext> context)
        : Runtime_(std::move(context))
    {
        InitStreams();
        InitTask();
    }

    TLogStreamManager::~TLogStreamManager() {
        CheckRuntimeTask_.reset();
        {
            std::unique_lock lock(StreamsLock_);
            Streams_.clear();
        }
    }

    void TLogStreamManager::InitStreams() {
        TLog::Debug() << "LogManager: Initializing log streams";
        std::unique_lock lock(StreamsLock_);
        Streams_.clear();
        StreamContext_.Set(std::make_shared<TContext>(Runtime_->PrepareContext()));

        for (const auto& [filename, configs] : StreamContext_.Get()->Configs) {
            for (const auto& [hostname, config] : configs) {
                NUnistat::TNameFactory::TTags tags = {
                    {"src_file", filename},
                    {"dst_host", hostname},
                    {"src_env", config.Ins.Common.Env},
                };

                Streams_.push_back(std::make_unique<TLogStream>(config, tags));
                TLog::Debug() << "LogManager: Initialized log stream for " << config.StreamId;
            }
        }
    }

    void TLogStreamManager::InitTask() {
        CheckRuntimeTask_ = std::make_unique<NUtils::TRegularTask>(
            [this]() {
                if (Runtime_->CheckContext(StreamContext_.Get())) {
                    return;
                }
                InitStreams();
            },
            Runtime_->GetFilesRescanPeriod(),
            "check_files_task");
    }

    void TLogStreamManager::AddUnistat(NUnistat::TBuilder& builder) {
        std::shared_lock lock(StreamsLock_);
        for (auto& stream : Streams_) {
            stream->AddUnistat(builder);
        }
        StreamContext_.Get()->Ctx->AddUnistat(builder);
    }

    NJuggler::TStatus TLogStreamManager::GetJugglerStatus() {
        NJuggler::TStatus res;
        std::shared_lock lock(StreamsLock_);
        for (auto& stream : Streams_) {
            res.Update(stream->GetJugglerStatus());
        }

        return res;
    }
}
