#include "logstoreagent.h"

#include "log_stream_manager.h"

#include <passport/infra/daemons/logstoreagent/src/cfg/runtime_context.h>

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/tvmauth/client/facade.h>

namespace NPassport::NLogstoreAgent {
    TLogstoreAgent::TLogstoreAgent() = default;

    TLogstoreAgent::~TLogstoreAgent() {
        StreamManager_.reset();
    }

    void TLogstoreAgent::Init(const NJson::TConfig& config) {
        TRuntimeContext::InitLogger(config.As<TString>("/logger/file"));
        Runtime_ = std::make_shared<TRuntimeContext>(config, "/service");
        StreamManager_ = std::make_unique<TLogStreamManager>(Runtime_);
    }

    static const TString HEADER_NAME_CONTENT_TYPE = "Content-Type";
    static const TString HEADER_CONTENT_TYPE_JSON = "application/json";
    static const TString HEADER_CONTENT_TYPE_TEXT = "text/plain";

    static TString SerializeError(TStringBuf msg) {
        TString res;
        NJson::TWriter wr(res);
        NJson::TObject root(wr);

        root.Add("status", "error");
        root.Add("message", msg);

        return res;
    }

    static void HandleError(NCommon::TRequest& req, HttpCodes code, TStringBuf msg) {
        TLog::Debug() << "Local HTTP error: [" << (int)code << "] " << msg;

        req.SetStatus(code);
        req.SetHeader(HEADER_NAME_CONTENT_TYPE, HEADER_CONTENT_TYPE_JSON);
        req.Write(SerializeError(msg));
    }

    void TLogstoreAgent::HandleRequest(NCommon::TRequest& req) {
        try {
            if (req.GetPath() == "/healthcheck") {
                HandleHealthCheck(req);
                return;
            }

            HandleError(req,
                        HTTP_NOT_FOUND,
                        NUtils::CreateStr("Path '", req.GetPath(), "' is unknown"));
        } catch (const std::exception& e) {
            HandleError(req, HTTP_INTERNAL_SERVER_ERROR, e.what());
        }
    }

    void TLogstoreAgent::HandleHealthCheck(NCommon::TRequest& req) {
        req.SetStatus(HTTP_OK);
        req.SetHeader(HEADER_NAME_CONTENT_TYPE, HEADER_CONTENT_TYPE_TEXT);
        req.Write(GetJugglerStatus());
    }

    void TLogstoreAgent::AddUnistat(NUnistat::TBuilder& builder) {
        StreamManager_->AddUnistat(builder);
    }

    NJuggler::TStatus TLogstoreAgent::GetJugglerStatus() {
        NJuggler::TStatus res;
        res.Update(Runtime_->Tvm()->GetStatus());
        res.Update(StreamManager_->GetJugglerStatus());

        return res;
    }
}
