#include "logstoreapi_stream.h"

#include "processor.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/generic/hash.h>
#include <util/string/cast.h>

#include <variant>

namespace NPassport::NLogstoreAgent {
    static const TString HEADER_CONTENT_ENCODING = "X-Ya-Content-Encoding";

    static const THashMap<TCommonConfig::ECompressionCodec, TString> ENCODINGS_NAMES = {
        {TCommonConfig::ECompressionCodec::GZip, "gzip"},
        {TCommonConfig::ECompressionCodec::PlainText, "plaintext"},
        {TCommonConfig::ECompressionCodec::ZStd0, "zstd"},
        {TCommonConfig::ECompressionCodec::ZStd3, "zstd"},
        {TCommonConfig::ECompressionCodec::ZStd5, "zstd"},
    };

    const TString& TLogstoreApiStream::GetEncodingName(TCommonConfig::ECompressionCodec codec) {
        Y_ENSURE(ENCODINGS_NAMES.contains(codec), "LogstoreApiClient: Unknown compression type '" << int(codec) << "'");
        return ENCODINGS_NAMES.at(codec);
    }

    static const THashMap<TCommonConfig::ETimeAggregation, TString> AGGREGATION_NAMES = {
        {TCommonConfig::ETimeAggregation::Day, "day"},
        {TCommonConfig::ETimeAggregation::Hour, "hour"},
    };

    const TString& TLogstoreApiStream::GetAggregationName(TCommonConfig::ETimeAggregation aggregation) {
        Y_ENSURE(AGGREGATION_NAMES.contains(aggregation), "LogstoreApiClient: Unknown aggregation type '" << int(aggregation) << "'");
        return AGGREGATION_NAMES.at(aggregation);
    }

    static std::variant<size_t, TString> GetOffsetFromJson(const TString& body) {
        rapidjson::Document document;
        size_t offset;
        if (!NJson::TReader::DocumentAsObject(body, document)) {
            return "Invalid json body";
        }
        if (!NJson::TReader::MemberAsUInt64(document, "offset", offset)) {
            return "Invalid offset";
        }
        return offset;
    }

    TLogstoreApiStream::TMissingDataException::TMissingDataException(size_t offset)
        : ExpectedOffset(offset)
    {
    }

    TLogstoreApiStream::TMissingDataException TLogstoreApiStream::TMissingDataException::CreateFrom(NDbPool::THttpResponse& response) {
        auto offsetOrError = GetOffsetFromJson(response.Body);
        if (const auto* err = std::get_if<TString>(&offsetOrError)) {
            return TMissingDataException()
                   << "[" << response.Status << "] " << NUtils::EscapeEol(response.Body)
                   << ". Failed to parse expected offset: " << *err;
        }
        return TMissingDataException(std::get<size_t>(offsetOrError))
               << "[" << response.Status << "] " << NUtils::EscapeEol(response.Body);
    }

    TLogstoreApiStream::TLogstoreApiStream(std::shared_ptr<NDbPool::TDbPool> pool,
                                           TLogstoreApiStreamSettings&& settings)
        : Pool_(pool)
        , Settings_(std::move(settings))
        , Codec_(GetEncodingName(Settings_.Codec))
        , Aggregation_(GetAggregationName(Settings_.Aggregation))
    {
        Y_ENSURE(Pool_, "Pool must be initialized");
    }

    void TLogstoreApiStream::Start(TInstant createTime, size_t iNode, size_t offset) {
        Stop();

        TLog::Debug() << "LogstoreApiStream " << Settings_.ClientId
                      << " starting for inode=" << iNode << ";offset=" << offset << ";create_time=" << createTime;

        TString streamId = ToString(iNode);

        NDbPool::TBlockingHandle h(*Pool_);
        NDbPool::THttpResponse response = h.Query(MakeCreateRequest(createTime, streamId, offset))->ToHttpResponse();
        switch (response.Status) {
            case HTTP_OK:
                break;
            case HTTP_UNPROCESSABLE_ENTITY:
                throw TMissingDataException::CreateFrom(response);
            default:
                throw yexception() << "[" << response.Status << "] " << NUtils::EscapeEol(response.Body);
        }

        StreamId_ = std::move(streamId);

        TLog::Debug() << "LogstoreApiStream " << Settings_.ClientId << " started. StreamId=" << StreamId_.value();
    }

    void TLogstoreApiStream::Stop() {
        StreamId_.reset();
    }

    TString TLogstoreApiStream::PrepareChunk(const TString& chunk) const {
        return TCompressionProcessor::Compress(chunk, Settings_.Codec);
    }

    void TLogstoreApiStream::Push(TString&& chunk, size_t offset) {
        Y_ENSURE(IsValid(), "Stream is not valid");

        NDbPool::TNonBlockingHandle h(*Pool_);
        h.SendQuery(MakePushRequest(std::move(chunk), StreamId_.value(), offset));
        Handles_.emplace_back(std::move(h));
    }

    TLogstoreApiStream::TResult TLogstoreApiStream::Wait() {
        std::vector<NDbPool::TNonBlockingHandle> queueCopy(std::move(Handles_));

        TResult res;
        for (auto& handle : queueCopy) {
            NDbPool::THttpResponse response;
            try {
                response = handle.WaitResult()->ToHttpResponse();
            } catch (const std::exception& e) {
                res.Error = NUtils::CreateStr("failed to wait response: ", e.what());
                return res;
            }

            switch (response.Status) {
                case HTTP_OK:
                    res.Completed = ++res.Accepted;
                    break;
                case HTTP_ACCEPTED:
                    ++res.Accepted;
                    TLog::Debug() << "LogstoreApiStream " << Settings_.ClientId << ": chunk was accepted";
                    break;
                case HTTP_UNPROCESSABLE_ENTITY:
                    Stop();
                    throw TMissingDataException::CreateFrom(response);
                default:
                    Stop();
                    res.Error = NUtils::CreateStr("[", response.Status, "] ", NUtils::EscapeEol(response.Body));
                    return res;
            }
        }

        return res;
    }

    bool TLogstoreApiStream::IsValid() const {
        return StreamId_.has_value();
    }

    NDbPool::TQuery TLogstoreApiStream::MakeCreateRequest(TInstant createTime, const TString& streamId, size_t offset) const {
        return NDbPool::TQuery(NUtils::CreateStr(
            "/1/create_stream",
            "?file=", Settings_.File,
            "&host=", Settings_.Host,
            "&env=", Settings_.Env,
            "&aggregation=", Aggregation_,
            "&create_time=", createTime.Seconds(),
            "&stream_id=", streamId,
            "&offset=", offset));
    }

    NDbPool::TQuery TLogstoreApiStream::MakePushRequest(TString&& chunk, const TString& streamId, size_t offset) const {
        NDbPool::TQuery query(NUtils::CreateStr(
            "/1/push",
            "?file=", Settings_.File,
            "&host=", Settings_.Host,
            "&env=", Settings_.Env,
            "&stream_id=", streamId,
            "&offset=", offset));

        query.SetHttpMethod("POST");
        query.SetHttpBody(std::move(chunk));
        query.AddHttpHeader(HEADER_CONTENT_ENCODING, Codec_);

        return query;
    }
}
