#include "scheduler.h"

#include "logstoreapi_stream.h"

#include <passport/infra/daemons/logstoreagent/src/utils/throttled_state_holder.h>

#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/tail/reader.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <functional>

namespace NPassport::NLogstoreAgent {
    TScheduler::TScheduler(std::unique_ptr<ILogstoreApiStream> httpStream,
                           TSchedulerSettings&& settings,
                           const NUnistat::TNameFactory::TTags& unistatTags)
        : HttpStream_(std::move(httpStream))
        , Settings_(std::move(settings))
        , AgeOldestInQueue_(TInstant::Max())
        , AgeOldestInBuffer_(TInstant::Max())
    {
        Y_ENSURE(HttpStream_, "Stream must be initialized");

        StateHolder_ = std::make_unique<TThrottledStateHolder>(Settings_.CacheDir);
        InitUnistat(unistatTags);
    }

    TScheduler::~TScheduler() = default;

    bool TScheduler::TryPush(TChunkPtr chunk, TDuration timeout) {
        if (!WaitBuffer(timeout)) {
            return false;
        }

        TInstant now(TInstant::Now());
        AgeOldestInBuffer_ = now;

        chunk->Data = HttpStream_->PrepareChunk(chunk->Data);

        std::unique_lock lock(BufferLock_);
        Buffer_.push_back(TChunkHolder{
            .Chunk = std::move(chunk),
            .Timestamp = now,
        });

        return true;
    }

    ETaskStatus TScheduler::Run() {
        LogHeartbeat();
        PopulateQueueFromBuffer();

        if (Queue_.empty()) {
            AgeOldestInQueue_ = TInstant::Max();
            return ETaskStatus::Idle;
        }

        AgeOldestInQueue_ = Queue_.front().Timestamp;

        if (!IsStreamValid()) {
            CreateStream();
        }

        PushChunks();

        ILogstoreApiStream::TResult result;
        try {
            result = HttpStream_->Wait();
        } catch (const ILogstoreApiStream::TMissingDataException& e) {
            StateHolder_->Write(e.ExpectedOffset);
            throw TFatalException() << "Scheduler " << Settings_.SchedulerId
                                    << ": Failed to push: " << e.what()
                                    << ". Offset reset to: " << e.ExpectedOffset;
        }

        ChunksSent_ += result.Accepted;

        if (result.Completed != 0) {
            auto lastSuccess = Queue_.begin();
            std::advance(lastSuccess, result.Completed - 1);
            StateHolder_->Write(lastSuccess->Chunk->ROffset);
            Queue_.erase(Queue_.begin(), std::next(lastSuccess));
        }

        if (result.Error) {
            TLog::Warning() << "Scheduler " << Settings_.SchedulerId
                            << ": Failed to push: " << *result.Error
                            << ". retryable=" << HttpStream_->IsValid();
            return ETaskStatus::Failure;
        }

        return result.Completed == 0 ? ETaskStatus::Failure : ETaskStatus::Success;
    }

    IStateHolder::TState TScheduler::GetState() {
        std::optional<IStateHolder::TState> state = StateHolder_->Read();
        return (state ? *state : IStateHolder::TState{});
    }

    NJuggler::TStatus TScheduler::GetJugglerStatus() const {
        const TDuration elapsed = TInstant::Now() - std::min(AgeOldestInQueue_.load(), AgeOldestInBuffer_.load());

        auto buildMessage = [&]() {
            return NUtils::CreateStr(
                "No chunks were sent for ", elapsed.Minutes(), " min: ", Settings_.SchedulerId);
        };

        if (elapsed > TDuration::Minutes(30)) {
            return NJuggler::TStatus(NJuggler::ECode::Critical, buildMessage());
        }

        if (elapsed > TDuration::Minutes(15)) {
            return NJuggler::TStatus(NJuggler::ECode::Warning, buildMessage());
        }

        return {};
    }

    void TScheduler::AddUnistat(NUnistat::TBuilder& builder) {
        std::shared_lock lock(BufferLock_);
        builder.AddRow(SignalBufferSizeName_, Buffer_.size());
        builder.AddRow(SignalQueueSizeName_, Queue_.size());
    }

    void TScheduler::InitUnistat(const NUnistat::TNameFactory::TTags& tags) {
        NUnistat::TNameFactory helper("scheduler", tags);

        SignalBufferSizeName_ = helper.Name("buffer_size", NUnistat::NSuffix::AXXX);
        SignalQueueSizeName_ = helper.Name("queue_size", NUnistat::NSuffix::AXXX);
    }

    bool TScheduler::WaitBuffer(TDuration timeout) {
        TInstant deadline = timeout.ToDeadLine();
        while (true) {
            {
                std::shared_lock lock(BufferLock_);
                if (Buffer_.size() < Settings_.BufferSize) {
                    return true;
                }
            }

            if (!BufferEvent_.WaitD(deadline)) {
                return false;
            }
        }
    }

    void TScheduler::CreateStream() {
        Y_ENSURE(!Queue_.empty(), "Cannot create stream without chunks");

        IStateHolder::TState state = {
            .Offset = Queue_.front().Chunk->Offset,
            .Inode = Queue_.front().Chunk->INode,
        };

        try {
            HttpStream_->Start(
                Queue_.front().Chunk->CreateTime,
                state.Inode,
                state.Offset);
        } catch (const ILogstoreApiStream::TMissingDataException& e) {
            state.Offset = e.ExpectedOffset;
            StateHolder_->Write(state);
            throw TFatalException() << "Scheduler " << Settings_.SchedulerId
                                    << ": Failed to create stream: " << e.what()
                                    << ". Offset reset to: " << e.ExpectedOffset;
        } catch (const std::exception& e) {
            throw yexception() << "Scheduler " << Settings_.SchedulerId
                               << ": Failed to create stream: " << e.what();
        }

        StateHolder_->Write(state);

        // empty chunk can't be written to API
        if (Queue_.front().Chunk->IsEmpty) {
            Queue_.pop_front();
        }
    }

    bool TScheduler::IsStreamValid() {
        Y_ENSURE(!Queue_.empty(), "Cannot create stream without chunks");

        std::optional<IStateHolder::TState> state = StateHolder_->Read();
        if (!state || !HttpStream_->IsValid()) {
            return false;
        }

        const size_t nextINode = Queue_.front().Chunk->INode;
        if (nextINode != state->Inode) {
            TLog::Debug() << "Scheduler " << Settings_.SchedulerId
                          << ": inode changed while processing queue (prev: " << state->Inode
                          << ", next: " << nextINode << ")";
            return false;
        }

        return true;
    }

    void TScheduler::PushChunks() {
        std::optional<IStateHolder::TState> state = StateHolder_->Read();
        Y_ENSURE(state, "State cannot be empty");

        size_t sentCount = 0;
        for (auto it = Queue_.begin(); it != Queue_.end(); ++it) {
            if (it->Chunk->INode != state->Inode) {
                TLog::Debug() << "Scheduler " << Settings_.SchedulerId
                              << ": inode changed while sending requests (prev: " << state->Inode
                              << ", next: " << it->Chunk->INode
                              << "). So we are trying to send only " << sentCount
                              << " of " << Queue_.size() << " chunks";
                break;
            }
            ++sentCount;

            try {
                HttpStream_->Push(TString(it->Chunk->Data), it->Chunk->Offset);
            } catch (const NDbPool::TException& e) {
                TLog::Warning() << "Scheduler " << Settings_.SchedulerId << ": cannot send chunk: " << e.what();
                break;
            }
        }

        ChunksTriesToSend_ += sentCount;
    }

    void TScheduler::PopulateQueueFromBuffer() {
        {
            std::unique_lock lock(BufferLock_);
            auto it = Buffer_.begin();
            std::advance(it, std::min(Settings_.MaxQueueSize - std::min(Queue_.size(), Settings_.MaxQueueSize), Buffer_.size()));
            Queue_.splice(Queue_.end(), Buffer_, Buffer_.begin(), it);
            AgeOldestInBuffer_ = Buffer_.empty() ? TInstant::Max() : Buffer_.front().Timestamp;
        }

        BufferEvent_.Signal();
    }

    void TScheduler::LogHeartbeat() {
        const TInstant now = TInstant::Now();
        if (now < LastHeartbeat_ + TDuration::Minutes(5)) {
            return;
        }
        LastHeartbeat_ = now;

        TLog::Debug() << "Scheduler " << Settings_.SchedulerId
                      << ": heartbeat. Chunks sent: " << ChunksSent_
                      << ". Attempts to send: " << ChunksTriesToSend_;
    }
}
