#include "backoff.h"

#include <util/random/random.h>

namespace NPassport::NLogstoreAgent {
    TBackoff::TBackoff(TBackoffSettings settings)
        : Settings_(settings)
        , Timeout_(Settings_.MinTimeout)
    {
        Y_ENSURE(Settings_.MinTimeout, "Minimum timeout must be specified");
        Y_ENSURE(Settings_.Policy == EBackoffPolicy::Constant || Settings_.MaxTimeout, "Maximum timeout must be specified, unless policy = Constant");
        if (!Settings_.MaxTimeout) {
            Settings_.MaxTimeout = Settings_.MinTimeout;
        }
        Y_ENSURE(Settings_.MinTimeout <= Settings_.MaxTimeout, "Max timeout cannot be smaller than min timeout");
    }

    TDuration TBackoff::GetTimeout() const {
        return Normalize(Jitter(Timeout_));
    }

    void TBackoff::operator++() {
        switch (Settings_.Policy) {
            case EBackoffPolicy::Constant:
                break;
            case EBackoffPolicy::Linear:
                Timeout_ += Settings_.MinTimeout;
                break;
            case EBackoffPolicy::Exponential:
                Timeout_ *= 2;
                break;
        }
        Timeout_ = Normalize(Timeout_);
    }

    void TBackoff::Reset() {
        Timeout_ = Settings_.MinTimeout;
    }

    TDuration TBackoff::Jitter(const TDuration timeout) const {
        if (!Settings_.Jitter) {
            return timeout;
        }

        ui64 ls = std::min(timeout - Settings_.MinTimeout, Settings_.Jitter).MicroSeconds();
        ui64 rs = std::min(Settings_.MaxTimeout - timeout, Settings_.Jitter).MicroSeconds();
        return timeout + TDuration::MicroSeconds(RandomNumber(ls + rs)) - TDuration::MicroSeconds(ls);
    }

    TDuration TBackoff::Normalize(const TDuration timeout) const {
        return std::min(std::max(timeout, Settings_.MinTimeout), Settings_.MaxTimeout);
    }
}
