#include "state_holder.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/stream/file.h>

namespace NPassport::NLogstoreAgent {
    TDiskStateHolder::TDiskStateHolder(const TFsPath& dir)
        : Lock_(dir)
        , State_(dir / "state", OpenAlways | RdWr)
    {
    }

    std::optional<TDiskStateHolder::TState> TDiskStateHolder::Read() const {
        try {
            // Reopen file to seek 0
            TFileInput in(State_.GetName());
            return Parse(in.ReadAll());
        } catch (const std::exception& e) {
            TLog::Warning() << "DiskStateHolder: failed to read state from " << State_.GetName()
                            << ": " << e.what();
        }
        return {};
    }

    void TDiskStateHolder::Write(const TState& state) {
        State_.Seek(0, sSet);

        const TString buf = Serialize(state);
        State_.Write(buf.data(), buf.size());
        State_.Flush();
    }

    TDiskStateHolder::TState TDiskStateHolder::Parse(TStringBuf body) {
        Y_ENSURE(!body.empty(),
                 "There is no state");

        ui8 version = body[0];
        Y_ENSURE(version == 1,
                 "version is not supported: " << (ui32)version);

        const size_t expectedSize = 1 + sizeof(TState::Inode) + sizeof(TState::Offset);
        static_assert(expectedSize == 17, "");
        Y_ENSURE(body.size() == expectedSize,
                 "bad body size: expected " << expectedSize << ", got " << body.size());

        TState res;
        res.Offset = NUtils::FromBytesMsb<ui64>(body, 1);
        res.Inode = NUtils::FromBytesMsb<ui64>(body, 9);
        return res;
    }

    TString TDiskStateHolder::Serialize(const TState& state) {
        TString res(17, 0);

        res[0] = 1;
        NUtils::ToBytesMsb(state.Offset, res, 1);
        NUtils::ToBytesMsb(state.Inode, res, 9);

        return res;
    }

    bool IStateHolder::TState::operator==(const TState& o) const {
        return Offset == o.Offset && Inode == o.Inode;
    }
}
