#pragma once

#include "backoff.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/threading/future/future.h>

#include <util/system/event.h>
#include <util/system/thread.h>

#include <thread>

namespace NPassport::NLogstoreAgent {
    class TFatalException: public yexception {
    };

    struct TTaskSettings {
        TBackoffSettings BackoffSettings;
        TString Name;
    };

    enum class ETaskStatus {
        Success,
        Idle,
        Failure,
    };

    template <class T>
    class TTaskDecorator: TNonCopyable, public T {
    public:
        template <typename... VarArgs>
        TTaskDecorator(const TTaskSettings& settings,
                       VarArgs&&... varArgs)
            : T(std::forward<VarArgs>(varArgs)...)
            , Backoff_(std::make_unique<TBackoff>(settings.BackoffSettings))
            , Shutdown_(false)
            , ShutdownPromise_(NThreading::NewPromise<void>())
        {
            Worker_ = std::thread([this, name = settings.Name]() { Execute(name); });
        }

        ~TTaskDecorator() override {
            try {
                Shutdown();
            } catch (const std::exception& e) {
                TLog::Warning() << "TaskDecorator: failed to shutdown: " << e.what();
            }
        }

        NThreading::TFuture<void> ShutdownHook() {
            return ShutdownPromise_.GetFuture();
        }

        void Shutdown() {
            Shutdown_ = true;
            Event_.Signal();
            if (Worker_.joinable()) {
                Worker_.join();
            }
            ShutdownPromise_.TrySetValue();
        }

        void RunNow() {
            Event_.Signal();
        }

        template <typename... TArgs>
        static std::unique_ptr<TTaskDecorator<T>> CreateUnique(const TTaskSettings& settings, TArgs&&... args) {
            return std::make_unique<TTaskDecorator<T>>(settings, std::forward<TArgs>(args)...);
        }

    private:
        void Execute(const TString name) {
            if (!name.empty()) {
                TThread::SetCurrentThreadName(name.data());
            }

            ETaskStatus previousStatus = ETaskStatus::Idle;
            while (!Shutdown_.load(std::memory_order_relaxed)) {
                ETaskStatus status;
                try {
                    status = T::Run();
                } catch (const TFatalException& e) {
                    TLog::Error() << "Unfixable error in " << name << ": " << e.what();
                    ShutdownPromise_.TrySetValue();
                    break;
                } catch (const std::exception& e) {
                    TLog::Error() << "Task cycle failed in " << name << ": " << e.what();
                    status = ETaskStatus::Failure;
                }

                switch (status) {
                    case ETaskStatus::Success:
                        break;
                    case ETaskStatus::Idle:
                    case ETaskStatus::Failure:
                        Event_.WaitT(Backoff_->GetTimeout());
                        if (previousStatus == status) {
                            ++*Backoff_;
                        }
                        break;
                }

                if (previousStatus != status) {
                    Backoff_->Reset();
                }
                previousStatus = status;
            }
        };

    private:
        std::unique_ptr<TBackoff> Backoff_;

        std::atomic_bool Shutdown_;
        std::thread Worker_;
        TAutoEvent Event_;
        NThreading::TPromise<void> ShutdownPromise_;
    };
}
