#include <passport/infra/daemons/logstoreagent/src/output/logstoreapi_stream.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NLogstoreAgent;

Y_UNIT_TEST_SUITE(LogstoreApiStream) {
    class TTestLogstoreApiStream: public TLogstoreApiStream {
    public:
        using TLogstoreApiStream::GetAggregationName;
        using TLogstoreApiStream::GetEncodingName;
        using TLogstoreApiStream::MakeCreateRequest;
        using TLogstoreApiStream::MakePushRequest;
        using TLogstoreApiStream::TLogstoreApiStream;
    };

    static TTestLogstoreApiStream MakeTestStream(
        TCommonConfig::ETimeAggregation aggregation = TCommonConfig::ETimeAggregation::Hour) {
        NDbPool::TDbPoolSettings s{
            .Dsn = NDbPool::TDestination::CreateHttp("some_host"),
            .DefaultQueryOpts = std::make_shared<NDbPool::TQueryOpts>(),
        };
        return TTestLogstoreApiStream(
            std::make_shared<NDbPool::TDbPool>(s),
            TLogstoreApiStreamSettings{
                .ClientId = "test_stream",
                .File = "some_file",
                .Host = "some_host",
                .Env = "testing",
                .Codec = TCommonConfig::ECompressionCodec::PlainText,
                .Aggregation = aggregation,
            });
    }

    Y_UNIT_TEST(Init) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLogstoreApiStream(nullptr, {}), yexception, "Pool must be initialized");

        auto stream = MakeTestStream();
        UNIT_ASSERT(!stream.IsValid());
        UNIT_ASSERT_EXCEPTION_CONTAINS(stream.Push("some chunk", 0), yexception, "Stream is not valid");
    }

    Y_UNIT_TEST(GetEncodingName) {
        UNIT_ASSERT_EQUAL(TTestLogstoreApiStream::GetEncodingName(TCommonConfig::ECompressionCodec::PlainText), "plaintext");
        UNIT_ASSERT_EQUAL(TTestLogstoreApiStream::GetEncodingName(TCommonConfig::ECompressionCodec::GZip), "gzip");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TTestLogstoreApiStream::GetEncodingName(TCommonConfig::ECompressionCodec(-1)),
            yexception,
            "LogstoreApiClient: Unknown compression type '-1'");
    }

    Y_UNIT_TEST(GetAggregationName) {
        UNIT_ASSERT_EQUAL(TTestLogstoreApiStream::GetAggregationName(TCommonConfig::ETimeAggregation::Day), "day");
        UNIT_ASSERT_EQUAL(TTestLogstoreApiStream::GetAggregationName(TCommonConfig::ETimeAggregation::Hour), "hour");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TTestLogstoreApiStream::GetAggregationName(TCommonConfig::ETimeAggregation(-1)),
            yexception,
            "LogstoreApiClient: Unknown aggregation type '-1'");
    }

    Y_UNIT_TEST(MakeCreateRequest) {
        auto req = MakeTestStream().MakeCreateRequest(TInstant::Seconds(1649082094), "666", 100500);
        UNIT_ASSERT_EQUAL(req.Query(), "/1/create_stream?file=some_file&host=some_host&env=testing&aggregation=hour&create_time=1649082094&stream_id=666&offset=100500");

        req = MakeTestStream(TCommonConfig::ETimeAggregation::Day).MakeCreateRequest(TInstant::Seconds(1649081155), "123", 9999);
        UNIT_ASSERT_EQUAL(req.Query(), "/1/create_stream?file=some_file&host=some_host&env=testing&aggregation=day&create_time=1649081155&stream_id=123&offset=9999");
    }

    Y_UNIT_TEST(MakePushRequest) {
        auto stream = MakeTestStream();

        auto req = stream.MakePushRequest("some/ndata/nin/nchunk/n", "666", 100500);
        UNIT_ASSERT_EQUAL(req.HttpMethod(), "POST");
        UNIT_ASSERT_EQUAL(req.Query(), "/1/push?file=some_file&host=some_host&env=testing&stream_id=666&offset=100500");
        UNIT_ASSERT_EQUAL(req.HttpBody(), "some/ndata/nin/nchunk/n");
    }

    Y_UNIT_TEST(CreateMissingDataException) {
        auto response = NDbPool::THttpResponse{
            .Status = HTTP_OK,
            .Body = R"({"offset": 129})",
        };

        auto e1 = TLogstoreApiStream::TMissingDataException::CreateFrom(response);
        UNIT_ASSERT_STRING_CONTAINS(TString(e1.what()), R"([200] {"offset": 129})");
        UNIT_ASSERT_EQUAL(e1.ExpectedOffset, 129);

        response.Body = "invalid json";
        auto e2 = TLogstoreApiStream::TMissingDataException::CreateFrom(response);
        UNIT_ASSERT_STRING_CONTAINS(TString(e2.what()), "[200] invalid json. Failed to parse expected offset: Invalid json body");
        UNIT_ASSERT_EQUAL(e2.ExpectedOffset, 0);

        response.Body = R"({"offset": "129"})";
        auto e3 = TLogstoreApiStream::TMissingDataException::CreateFrom(response);
        UNIT_ASSERT_STRING_CONTAINS(TString(e3.what()), R"([200] {"offset": "129"}. Failed to parse expected offset: Invalid offset)");
        UNIT_ASSERT_EQUAL(e3.ExpectedOffset, 0);
    }
}
