#include <passport/infra/daemons/logstoreagent/src/output/logstoreapi_stream.h>
#include <passport/infra/daemons/logstoreagent/src/output/scheduler.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NLogstoreAgent;

Y_UNIT_TEST_SUITE(Scheduler) {
    static const TFsPath CACHE_DIR = GetOutputPath() / "scheduler_cache";

    static void PrepareCacheDir() {
        NFs::RemoveRecursive(CACHE_DIR);
        NFs::MakeDirectory(CACHE_DIR);
    }

    struct TStream {
        size_t InitINode = 0;
        size_t InitOffset = 0;
        TInstant CreateTime;
        std::vector<std::pair<size_t, TString>> ExpectedChunks;
    };

    using TTestCase = std::vector<TStream>;

    class TErrorHolder {
    public:
        TString GetErrorMsg() const {
            TString msg;
            for (const auto& error : Errors_) {
                NUtils::AppendSeparated(msg, '\n', error);
            }
            return msg;
        }

        bool IsOk() const {
            return Errors_.empty();
        }

        void SetError(const TString msg) {
            Errors_.push_back(msg);
        }

    private:
        std::vector<TString> Errors_;
    };

    class TFakeLogstoreApiStream: public ILogstoreApiStream {
    public:
        TFakeLogstoreApiStream(const TTestCase& testCase = {}, size_t softLimit = 2, std::optional<size_t> hardLimit = {}, std::optional<size_t> errorLimit = {})
            : TestCase_(testCase)
            , SoftLimit_(softLimit)
            , HardLimit_(hardLimit ? *hardLimit : SoftLimit_)
            , ErrorLimit_(errorLimit ? *errorLimit : HardLimit_)
            , CurrentStream_(TestCase_.cbegin())
            , Error_(std::make_shared<TErrorHolder>())
        {
        }

        ~TFakeLogstoreApiStream() override {
            try {
                Stop();
                if (CurrentStream_ != TestCase_.cend()) {
                    Error_->SetError(TStringBuilder() << "Some streams were not completed. Stopping at " << CurrentStream_->InitINode);
                }
            } catch (const std::exception& e) {
                TLog::Debug() << "~TFakeLogstoreApiStream: " << e.what();
            }
        }

        std::shared_ptr<TErrorHolder> GetErrorHolder() const {
            return Error_;
        }

        void Exception(const TString& msg) const {
            Error_->SetError(msg);
            ythrow yexception() << "Test failed with error: " << msg;
        }

        void Start(TInstant createTime, size_t iNode, size_t offset) override {
            Stop();

            if (CurrentStream_ == TestCase_.cend()) {
                Exception("Can't start next stream: All expected streams were completed");
            }

            if (createTime != CurrentStream_->CreateTime) {
                Exception(TStringBuilder() << "Wrong create time: got " << createTime << " expected " << CurrentStream_->CreateTime);
            }
            if (iNode != CurrentStream_->InitINode) {
                Exception(TStringBuilder() << "Wrong initial inode: got " << iNode << " expected " << CurrentStream_->InitINode);
            }
            if (offset != CurrentStream_->InitOffset) {
                Exception(TStringBuilder() << "Wrong initial offset: got " << offset << ", expected " << CurrentStream_->InitOffset);
            }

            IsValid_ = true;
        }

        void Stop() {
            if (!IsValid()) {
                return;
            }

            if (CompletedChunks_ != CurrentStream_->ExpectedChunks.size()) {
                Error_->SetError(TStringBuilder() << "Completed only " << CompletedChunks_
                                                  << " of total " << CurrentStream_->ExpectedChunks.size()
                                                  << " for streamId" << CurrentStream_->InitINode);
            }
            if (AcceptedChunks_ > 0) {
                Error_->SetError(TStringBuilder() << AcceptedChunks_ << " chunks are not completed"
                                                  << " for streamId" << CurrentStream_->InitINode);
            }

            CompletedChunks_ = 0;
            AcceptedChunks_ = 0;
            ++CurrentStream_;

            IsValid_ = false;
        }

        TString PrepareChunk(const TString& chunk) const override {
            return chunk + "_prepared_chunk";
        }

        void Push(TString&& chunk, size_t offset) override {
            if (!IsValid()) {
                Exception("Can't push: Stream not valid");
            }

            if (CompletedChunks_ + AcceptedChunks_ >= CurrentStream_->ExpectedChunks.size()) {
                Exception(TStringBuilder() << "Pushed more than expected: " << CurrentStream_->ExpectedChunks.size());
            }

            const auto& [expectedOffset, expectedChuck] =
                CurrentStream_->ExpectedChunks[CompletedChunks_ + AcceptedChunks_];

            if (offset != expectedOffset) {
                Exception(TStringBuilder() << "Wrong offset: got " << offset << ", expected " << expectedOffset);
            }
            if (chunk != PrepareChunk(expectedChuck)) {
                Exception(TStringBuilder() << "Wrong data: got " << chunk << ", expected " << PrepareChunk(expectedChuck));
            }

            if (++AcceptedChunks_ > HardLimit_) {
                Exception(TStringBuilder() << "Hard limit exceeded");
            }
        }

        TResult Wait() override {
            std::optional<TString> error;
            if (AcceptedChunks_ > ErrorLimit_) {
                AcceptedChunks_ = ErrorLimit_;
                error = "error forced by test";
            }

            TResult res = {
                .Completed = AcceptedChunks_ < SoftLimit_ ? AcceptedChunks_ : SoftLimit_,
                .Accepted = AcceptedChunks_,
                .Error = std::move(error),
            };

            CompletedChunks_ += res.Completed;
            AcceptedChunks_ = 0;

            return res;
        }

        bool IsValid() const override {
            return IsValid_;
        }

    private:
        const TTestCase TestCase_;
        const size_t SoftLimit_ = 2;
        const size_t HardLimit_ = 3;
        const size_t ErrorLimit_ = 3;

        size_t CompletedChunks_ = 0;
        size_t AcceptedChunks_ = 0;

        bool IsValid_ = false;
        TTestCase::const_iterator CurrentStream_;

        std::shared_ptr<TErrorHolder> Error_;
    };

    Y_UNIT_TEST(Init) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TScheduler(nullptr), yexception, "Stream must be initialized");
    }

    Y_UNIT_TEST(SimpleRun) {
        PrepareCacheDir();

        TTestCase testCase = {{
            .InitINode = 129,
            .InitOffset = 100500,
            .CreateTime = TInstant::Seconds(1649082094),
            .ExpectedChunks = {
                {100500, "chunk_1"},
                {100507, "chunk_2"},
            },
        }};

        std::shared_ptr<TErrorHolder> errorHolder;

        {
            auto stream = std::make_unique<TFakeLogstoreApiStream>(testCase, 2);
            errorHolder = stream->GetErrorHolder();
            auto scheduler = TScheduler(
                std::move(stream),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                    .SchedulerId = "test_scheduler",
                    .BufferSize = 2,
                    .MaxQueueSize = 2,
                });

            ETaskStatus status;
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 0);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 0);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100500,
                                              .ROffset = 100507,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_1",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100507,
                                              .ROffset = 100514,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_2",
                                              .IsEmpty = false,
                                          }),
                                          {}));

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100514);
        }

        UNIT_ASSERT_C(errorHolder->IsOk(), errorHolder->GetErrorMsg());

        {
            auto scheduler = TScheduler(
                std::make_unique<TFakeLogstoreApiStream>(),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                });

            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100514);
        }
    }

    Y_UNIT_TEST(LimitedRun) {
        PrepareCacheDir();

        TTestCase testCase = {{
            .InitINode = 129,
            .InitOffset = 100500,
            .CreateTime = TInstant::Seconds(1649082094),
            .ExpectedChunks = {
                {100500, "chunk_1"},
                {100507, "chunk_2"},
                {100514, "chunk_3"},
                {100521, "chunk_4"},
                {100528, "chunk_5"},
            },
        }};

        std::shared_ptr<TErrorHolder> errorHolder;

        {
            auto stream = std::make_unique<TFakeLogstoreApiStream>(testCase, 2, 3);
            errorHolder = stream->GetErrorHolder();
            auto scheduler = TScheduler(
                std::move(stream),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                    .SchedulerId = "test_scheduler",
                    .BufferSize = 4,
                    .MaxQueueSize = 3,
                });

            ETaskStatus status;
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 0);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 0);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100500,
                                              .ROffset = 100507,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_1",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100507,
                                              .ROffset = 100514,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_2",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100514,
                                              .ROffset = 100521,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_3",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100521,
                                              .ROffset = 100528,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_4",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(!scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                               .INode = 129,
                                               .Offset = 100528,
                                               .ROffset = 100535,
                                               .CreateTime = TInstant::Seconds(1649082094),
                                               .Data = "chunk_5",
                                               .IsEmpty = false,
                                           }),
                                           {}));

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100514);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100528,
                                              .ROffset = 100535,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_5",
                                              .IsEmpty = false,
                                          }),
                                          {}));

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100528);

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100535);

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100535);
        }

        UNIT_ASSERT_C(errorHolder->IsOk(), errorHolder->GetErrorMsg());

        {
            auto scheduler = TScheduler(
                std::make_unique<TFakeLogstoreApiStream>(),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                });

            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100535);
        }
    }

    Y_UNIT_TEST(EmptyLogRun) {
        PrepareCacheDir();

        TTestCase testCase = {{
            .InitINode = 129,
            .InitOffset = 100500,
            .CreateTime = TInstant::Seconds(1649082094),
            .ExpectedChunks = {},
        }};

        std::shared_ptr<TErrorHolder> errorHolder;

        {
            auto stream = std::make_unique<TFakeLogstoreApiStream>(testCase);
            errorHolder = stream->GetErrorHolder();
            auto scheduler = TScheduler(
                std::move(stream),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                    .SchedulerId = "test_scheduler",
                });

            ETaskStatus status;
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 0);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 0);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100500,
                                              .ROffset = 100500,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "",
                                              .IsEmpty = true,
                                          }),
                                          {}));
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Failure, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100500);
        }

        UNIT_ASSERT_C(errorHolder->IsOk(), errorHolder->GetErrorMsg());

        {
            auto scheduler = TScheduler(
                std::make_unique<TFakeLogstoreApiStream>(),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                });

            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100500);
        }
    }

    Y_UNIT_TEST(LogRotateRun) {
        PrepareCacheDir();

        TTestCase testCase = {
            {
                .InitINode = 129,
                .InitOffset = 100500,
                .CreateTime = TInstant::Seconds(1649082094),
                .ExpectedChunks = {
                    {100500, "chunk_1"},
                    {100507, "chunk_2"},
                },
            },
            {
                .InitINode = 666,
                .InitOffset = 300,
                .CreateTime = TInstant::Seconds(1649145862),
                .ExpectedChunks = {
                    {300, "chunk_3"},
                },
            },
            {
                .InitINode = 925,
                .InitOffset = 4532,
                .CreateTime = TInstant::Seconds(1649145979),
                .ExpectedChunks = {
                    {4532, "chunk_4"},
                },
            },
        };

        std::shared_ptr<TErrorHolder> errorHolder;

        {
            auto stream = std::make_unique<TFakeLogstoreApiStream>(testCase, 4);
            errorHolder = stream->GetErrorHolder();
            auto scheduler = TScheduler(
                std::move(stream),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                    .SchedulerId = "test_scheduler",
                    .BufferSize = 4,
                    .MaxQueueSize = 2,
                });

            ETaskStatus status;
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 0);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 0);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100500,
                                              .ROffset = 100507,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_1",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100507,
                                              .ROffset = 100514,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_2",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 666,
                                              .Offset = 300,
                                              .ROffset = 307,
                                              .CreateTime = TInstant::Seconds(1649145862),
                                              .Data = "chunk_3",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 925,
                                              .Offset = 4532,
                                              .ROffset = 4539,
                                              .CreateTime = TInstant::Seconds(1649145979),
                                              .Data = "chunk_4",
                                              .IsEmpty = false,
                                          }),
                                          {}));

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100514);

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 666);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 307);

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 925);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 4539);
        }

        UNIT_ASSERT_C(errorHolder->IsOk(), errorHolder->GetErrorMsg());

        {
            auto scheduler = TScheduler(
                std::make_unique<TFakeLogstoreApiStream>(),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                });

            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 925);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 4539);
        }
    }

    Y_UNIT_TEST(ErrorRun) {
        PrepareCacheDir();

        TTestCase testCase = {{
            .InitINode = 129,
            .InitOffset = 100500,
            .CreateTime = TInstant::Seconds(1649082094),
            .ExpectedChunks = {
                {100500, "chunk_1"},
                {100507, "chunk_2"},
                {100514, "chunk_3"},
            },
        }};

        std::shared_ptr<TErrorHolder> errorHolder;

        {
            auto stream = std::make_unique<TFakeLogstoreApiStream>(testCase, 2, 3, 2);
            errorHolder = stream->GetErrorHolder();
            auto scheduler = TScheduler(
                std::move(stream),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                    .SchedulerId = "test_scheduler",
                    .BufferSize = 4,
                    .MaxQueueSize = 3,
                });

            ETaskStatus status;
            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Idle, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 0);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 0);

            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100500,
                                              .ROffset = 100507,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_1",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100507,
                                              .ROffset = 100514,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_2",
                                              .IsEmpty = false,
                                          }),
                                          {}));
            UNIT_ASSERT(scheduler.TryPush(std::make_shared<TChunk>(TChunk{
                                              .INode = 129,
                                              .Offset = 100514,
                                              .ROffset = 100521,
                                              .CreateTime = TInstant::Seconds(1649082094),
                                              .Data = "chunk_3",
                                              .IsEmpty = false,
                                          }),
                                          {}));

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Failure, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100514);

            UNIT_ASSERT_NO_EXCEPTION_C(status = scheduler.Run(), errorHolder->GetErrorMsg());
            UNIT_ASSERT_EQUAL_C(status, ETaskStatus::Success, errorHolder->GetErrorMsg());
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100521);
        }

        UNIT_ASSERT_C(errorHolder->IsOk(), errorHolder->GetErrorMsg());

        {
            auto scheduler = TScheduler(
                std::make_unique<TFakeLogstoreApiStream>(),
                TSchedulerSettings{
                    .CacheDir = CACHE_DIR,
                });

            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Inode, 129);
            UNIT_ASSERT_VALUES_EQUAL(scheduler.GetState().Offset, 100521);
        }
    }
}
