#include "logger.h"

#include <passport/infra/libs/cpp/utils/thread_local_id.h>

#include <util/stream/output.h>
#include <util/string/cast.h>
#include <util/system/getpid.h>
#include <util/system/thread.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/uio.h>

namespace NPassport::NLogstoreApi {
    static const size_t RESERVE_COUNT = 10240;
    static const mode_t OPEN_MODE = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH;

    TLogger::TLogger(const TString& filename, const int linesPerShot)
        : Filename_(filename)
        , Pid_(IntToString<10>(GetPID()))
        , NeedReopen_(false)
        , Stopping_(false)
    {
        Queue_.reserve(RESERVE_COUNT);

        TString::size_type pos = 0;
        while (true) {
            pos = Filename_.find('/', pos + 1);
            if (TString::npos == pos) {
                break;
            }
            TString name = Filename_.substr(0, pos);
            int res = mkdir(name.c_str(), OPEN_MODE | S_IXUSR | S_IXGRP | S_IXOTH);
            if (-1 == res && EEXIST != errno) {
                Cerr << "failed to create dir: " << name << ". Errno: " << errno << Endl;
            }
        }

        Y_ENSURE(OpenFile(), "Failed to open file: " << Filename_);
        WritingThread_ = std::thread([this, linesPerShot]() {
            TThread::SetCurrentThreadName("logger");
            this->WritingThread(linesPerShot);
        });
    }

    TLogger::~TLogger() {
        Stopping_.store(true);
        QueueCondition_.notify_one();
        WritingThread_.join();

        if (Fd_ != -1) {
            close(Fd_);
        }
    }

    NThreading::TFuture<void> TLogger::Log(TString&& str) const {
        if (!str.EndsWith('\n')) {
            str.push_back('\n');
        }

        return AddToQueue(std::move(str));
    }

    void TLogger::Rotate() {
        NeedReopen_ = true;
    }

    bool TLogger::OpenFile() {
        if (Fd_ != -1) {
            close(Fd_);
        }
        Fd_ = open(Filename_.c_str(), O_WRONLY | O_CREAT | O_APPEND, OPEN_MODE);
        if (Fd_ == -1) {
            Cerr << "File logger cannot open file for writing: " << Filename_ << Endl;
        }
        return Fd_ != -1;
    }

    NThreading::TFuture<void> TLogger::AddToQueue(TString&& str) const {
        std::unique_lock lock(QueueMutex_);
        Queue_.emplace_back(TBuffer{
            .Data = std::move(str),
            .Promise = NThreading::NewPromise(),
        });

        // Trying to decrease csw counter
        // It is not necessary to notify for every row - huge logs cause too many system calls
        // Timers are comfort for human eyes and cheap for system
        auto now = TInstant::Now();
        if (now - LastNotify_ > TDuration::MilliSeconds(500)) {
            QueueCondition_.notify_one(); // system call
            LastNotify_ = now;
        }

        return Queue_.back().Promise.GetFuture();
    }

    void TLogger::WritingThread(const size_t linesPerShot) {
        std::vector<iovec> iovector;
        iovector.resize(linesPerShot);

        std::vector<TBuffer> queueCopy;
        queueCopy.reserve(RESERVE_COUNT);
        while (true) {
            queueCopy.clear();

            {
                std::unique_lock lock(QueueMutex_);
                if (Queue_.empty()) {
                    // timed_wait - protection against loss notify()
                    QueueCondition_.wait_for(lock, std::chrono::milliseconds(500));
                }
                std::swap(queueCopy, Queue_);
            }

            if (queueCopy.empty() && Stopping_.load(std::memory_order_relaxed)) {
                return;
            }

            if (NeedReopen_.load(std::memory_order_relaxed)) {
                NeedReopen_ = false;
                OpenFile();
            }

            if (Fd_ == -1) {
                for (TBuffer& s : queueCopy) {
                    s.Promise.SetException(std::make_exception_ptr(yexception() << "Bad file descriptor: " << Fd_ << " for " << Filename_));
                    Cerr << s.Data;
                }
                continue;
            }

            auto curStr = queueCopy.begin();
            while (curStr != queueCopy.end()) {
                const auto bulkStart = curStr;
                bool failed = false;
                ssize_t toWrite = 0;
                size_t endPos = 0;

                for (; endPos < iovector.size() && curStr != queueCopy.end(); ++endPos, ++curStr) {
                    iovector[endPos].iov_base = const_cast<char*>(curStr->Data.data());
                    iovector[endPos].iov_len = curStr->Data.size();
                    toWrite += curStr->Data.size();
                }

                size_t beginPos = 0;
                ssize_t writen = 0;
                while (writen < toWrite) {
                    ssize_t res = ::writev(Fd_, iovector.data() + beginPos, endPos - beginPos);
                    if (res < 0) {
                        failed = true;
                        Cerr << "Failed to write to log " << Filename_
                             << " : " << strerror(errno) << Endl;
                    } else {
                        writen += res;
                        if (writen >= toWrite) {
                            break;
                        }

                        while (static_cast<size_t>(res) >= iovector[beginPos].iov_len) {
                            res -= iovector[beginPos].iov_len;
                            ++beginPos;
                        }

                        if (res) {
                            iovector[beginPos].iov_len -= res;
                            iovector[beginPos].iov_base = static_cast<char*>(iovector[beginPos].iov_base) + res;
                        }
                    }
                }

                for (auto it = bulkStart; it != curStr; ++it) {
                    if (failed) {
                        it->Promise.SetException(std::make_exception_ptr(
                            yexception() << "Could not write to file: " << Filename_));
                    } else {
                        it->Promise.SetValue();
                    }
                }
            }
        }
    }
}
