#include "logstoreapi.h"

#include "utils/exceptions.h"
#include "utils/tz_msk.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/env.h>

namespace NPassport::NLogstoreApi {
    static const TString HEADER_CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_TYPE_JSON = "application/json";
    static const TString CONTENT_TYPE_TEXT = "text/plain";
    static const TString RESPONSE_OK = R"({"status": "OK", "error": ""})";
    static const TString RESPONSE_INTERNAL_ERROR = R"({"status": "ERROR", "error": "Internal error"})";
    static const TString HEADER_X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";

    static TString MakeJsonResponse(const TLogstoreApiError& exc) {
        TString response;

        NJson::TWriter wr(response);
        NJson::TObject root(wr);
        root.Add("status", "ERROR");
        root.Add("error", exc.Message());
        exc.AddFields(root);

        return response;
    }

    static void SetErrorStatus(NCommon::TRequest& req, HttpCodes code, const TLogstoreApiError& exc) {
        TLog::Warning() << "Request failed: " << req.GetQueryString() << ": " << exc.Message();

        req.SetStatus(code);
        req.SetHeader(HEADER_CONTENT_TYPE, CONTENT_TYPE_JSON);
        req.Write(MakeJsonResponse(exc));
    }

    TLogstoreApi::TLogstoreApi() = default;
    TLogstoreApi::~TLogstoreApi() = default;

    void TLogstoreApi::Init(const NJson::TConfig& config) {
        SetTzMsk();

        config.InitCommonLog("/logger");
        Processor_ = TProcessor::Create(config, "/service");
        InitTvm(config, "/service/tvm");
    }

    void TLogstoreApi::HandleRequest(NCommon::TRequest& req) {
        try {
            HandleRequestImpl(req);
        } catch (const TAuthError& e) {
            SetErrorStatus(req, HTTP_FORBIDDEN, e);
        } catch (const TBadRequestError& e) {
            SetErrorStatus(req, HTTP_BAD_REQUEST, e);
        } catch (const TBufferOverflowError& e) {
            SetErrorStatus(req, HTTP_CONFLICT, e);
        } catch (const TTimeoutError& e) {
            SetErrorStatus(req, HTTP_ACCEPTED, e);
        } catch (const TInvalidPathError& e) {
            SetErrorStatus(req, HTTP_NOT_FOUND, e);
        } catch (const TDuplicateError& e) {
            // FIXME: should not be ok, in theory
            SetErrorStatus(req, HTTP_OK, e);
        } catch (const TMissingDataError& e) {
            SetErrorStatus(req, HTTP_UNPROCESSABLE_ENTITY, e);
        } catch (const TLogstoreApiError& e) {
            SetErrorStatus(req, HTTP_INTERNAL_SERVER_ERROR, e);
        } catch (const std::exception& e) {
            TLog::Error() << "Uncaught error: " << e.what();

            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(RESPONSE_INTERNAL_ERROR);
        }
    }

    void TLogstoreApi::AddUnistat(NUnistat::TBuilder& builder) {
        Processor_->AddUnistat(builder);
    }

    void TLogstoreApi::AddUnistatExtended(const TString& path, NUnistat::TBuilder& builder) {
        TStringBuf p = path;
        p.ChopSuffix("/");

        if (p == "/logs") {
            Processor_->AddUnistatForLogs(builder);
        }
    }

    void TLogstoreApi::InitTvm(const NJson::TConfig& config, const TString& jpoint) {
        SelfTvmId_ = config.As<ui32>(jpoint + "/self_tvm_id");

        NTvmAuth::NTvmApi::TClientSettings settings;
        settings.SetSelfTvmId(SelfTvmId_);
        settings.EnableServiceTicketChecking();
        settings.SetDiskCacheDir(config.As<TString>(jpoint + "/disk_cache"));

        settings.SetTvmHostPort(
            config.As<TString>(jpoint + "/tvm_host", settings.TvmHost),
            config.As<ui32>(jpoint + "/tvm_port", settings.TvmPort));

        Tvm_ = std::make_unique<NTvmAuth::TTvmClient>(
            std::move(settings),
            NTvmLogger::TLogger::Create());

        std::vector<ui32> ids = config.As<std::vector<ui32>>(jpoint + "/allowed_tvmids");
        AllowedTvmIds_.insert(ids.begin(), ids.end());
    }

    void TLogstoreApi::HandleRequestImpl(NCommon::TRequest& req) {
        const TString& path = req.GetPath();

        if (path == "/ping") {
            req.SetStatus(HTTP_OK);
            req.Write("pong");
            return;
        }

        if (path == "/healthcheck") {
            req.SetStatus(HTTP_OK);
            req.SetHeader(HEADER_CONTENT_TYPE, CONTENT_TYPE_TEXT);
            req.Write(GetStatus());
            return;
        }

        if (path == "/1/push") {
            CheckAuth(req);
            Processor_->Process(req);

            req.SetStatus(HTTP_OK);
            req.Write(RESPONSE_OK);
            return;
        }

        if (path == "/1/create_stream") {
            CheckAuth(req);
            Processor_->CreateSession(req);

            req.SetStatus(HTTP_OK);
            req.Write(RESPONSE_OK);
            return;
        }

        throw TInvalidPathError() << path;
    }

    NJuggler::TStatus TLogstoreApi::GetStatus() const {
        NJuggler::TStatus res;

        res.Update(Tvm_->GetStatus());

        return res;
    }

    void TLogstoreApi::CheckAuth(const NCommon::TRequest& req) {
        const TString& header = req.GetHeader(HEADER_X_YA_SERVICE_TICKET);
        if (header.empty()) {
            throw TAuthError() << "missing header '" << HEADER_X_YA_SERVICE_TICKET << "'";
        }

        NTvmAuth::TCheckedServiceTicket ticket = Tvm_->CheckServiceTicket(header);
        if (ticket.GetStatus() != NTvmAuth::ETicketStatus::Ok) {
            throw TAuthError() << "ServiceTicket is invalid - " << ticket.GetStatus()
                               << " (" << ticket.DebugInfo()
                               << " - " << NTvmAuth::NUtils::RemoveTicketSignature(header) << ")"
                               << (ticket.GetStatus() == NTvmAuth::ETicketStatus::InvalidDst
                                       ? NUtils::CreateStr("; expected dst: ", SelfTvmId_)
                                       : TString());
        }

        if (!AllowedTvmIds_.contains(ticket.GetSrc())) {
            throw TAuthError() << "tvmid=" << ticket.GetSrc() << " is not allowed";
        }
    }
}
