#include "processor.h"

#include "parsers/plain_text_parser.h"
#include "utils/converter.h"
#include "utils/exceptions.h"
#include "writers/log_aggregator.h"
#include "writers/sequencer.h"
#include "writers/simple_writer.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

namespace NPassport::NLogstoreApi {
    TProcessor::TProcessor(std::unique_ptr<TLogAggregator> logAggregator, const TProcessorSettings& settings)
        : Aggregator_(std::move(logAggregator))
        , Settings_(settings)
    {
        Y_ENSURE(Aggregator_, "Log aggregator must be initialized");
    }

    TProcessor::~TProcessor() = default;

    void TProcessor::Process(const NCommon::TRequest& req) {
        TPushRequest pushRequest = TRequestConverter::ConvertPushRequest(req);
        SignalCompressedLogs_ += req.GetRequestBody().size();
        SignalRawLogs_ += pushRequest.Body.size();

        NThreading::TFuture<void> future = Aggregator_->Aggregate(Parse(pushRequest));
        if (!future.Wait(Settings_.RequestTimeout)) {
            throw TTimeoutError() << "Failed to write chunk in time";
        }
    }

    void TProcessor::CreateSession(const NCommon::TRequest& req) {
        TStreamRequest streamRequest = TRequestConverter::ConvertStreamRequest(req);
        Aggregator_->AddStream(streamRequest);
    }

    std::unique_ptr<TChunk> TProcessor::Parse(const TPushRequest& requestData) {
        // TODO: check content type and version, and map to parser?
        std::unique_ptr<TChunk> chunk = std::make_unique<TChunk>(
            requestData.StreamInfo,
            TPlainTextParser(requestData.Offset).Parse(requestData.Body));

        if (chunk->empty() && !requestData.Body.empty()) {
            throw TBadRequestError() << "Invalid chunk data";
        }

        return chunk;
    }

    std::unique_ptr<TProcessor> TProcessor::Create(const NJson::TConfig& config, const TString& point) {
        TProcessorSettings settings{
            .RequestTimeout = TDuration::MilliSeconds(config.As<ui64>(point + "/request_timeout__msecs")),
        };

        return std::make_unique<TProcessor>(
            TLogAggregator::Create(config, point),
            settings);
    }

    void TProcessor::AddUnistat(NUnistat::TBuilder& builder) {
        Aggregator_->AddUnistat(builder);
        builder.Add(SignalCompressedLogs_);
        builder.Add(SignalRawLogs_);
    }

    void TProcessor::AddUnistatForLogs(NUnistat::TBuilder& builder) {
        Aggregator_->AddUnistatForLogs(builder);
    }
}
