#include "converter.h"

#include "exceptions.h"
#include "extractor.h"

#include <library/cpp/http/misc/http_headers.h>

#include <util/generic/hash.h>

static const TString HEADER_CONTENT_TYPE = "X-Ya-Content-Type";
static const TString HEADER_CONTENT_ENCODING = "X-Ya-Content-Encoding";

static const TString ARG_ENV = "env";
static const TString ARG_FILE = "file";
static const TString ARG_SOURCE_HOST = "host";
static const TString ARG_STREAM_ID = "stream_id";
static const TString ARG_CREATION_TIME = "create_time";
static const TString ARG_OFFSET = "offset";
static const TString ARG_AGGREGATION = "aggregation";

namespace NPassport::NLogstoreApi {
    static const THashMap<TString, ECompressionCodec> ENCODING_MAP{
        {"", ECompressionCodec::PlainText},
        {"plaintext", ECompressionCodec::PlainText},
        {"gzip", ECompressionCodec::GZip},
        {"zstd", ECompressionCodec::ZStd},
    };

    static const TString& GetArg(const NCommon::TRequest& req, const TString& arg, bool required = false) {
        const TString& s = req.GetArg(arg);
        if (required && s.empty()) {
            throw TBadRequestError() << "Required parameter '" << arg << "' is missing";
        }
        return s;
    }

    template <class T>
    static T GetArg(const NCommon::TRequest& req, const TString& arg, bool required = false, const T& def = {}) {
        const TString& s = GetArg(req, arg, required);
        if (s.empty()) {
            return def;
        }

        T out;
        if (!TryFromString<T>(s, out)) {
            throw TBadRequestError() << "Parameter '" << arg << "' is invalid";
        }

        return out;
    }

    TStreamInfo GetStreamInfo(const NCommon::TRequest& req) {
        TStreamInfo info{
            .File = GetArg(req, ARG_FILE, true),
            .Host = GetArg(req, ARG_SOURCE_HOST, true),
            .Env = GetArg(req, ARG_ENV, true),
            .StreamId = GetArg(req, ARG_STREAM_ID, true),
        };

        info.File.SkipPrefix("/");
        return info;
    }

    TPushRequest TRequestConverter::ConvertPushRequest(const NCommon::TRequest& req) {
        const TString& encoding = req.GetHeader(HEADER_CONTENT_ENCODING);

        // TODO: maybe do this in create stream once
        auto it = ENCODING_MAP.find(encoding);
        if (it == ENCODING_MAP.end()) {
            throw TBadRequestError() << "Unknown encoding: " << encoding;
        }

        return TPushRequest{
            .StreamInfo = GetStreamInfo(req),
            .Offset = GetArg<size_t>(req, ARG_OFFSET, true),
            // TODO: make this a non-static member
            .Body = TCompressionProcessor(it->second).Extract(req.GetRequestBody()),
        };
    }

    TStreamRequest TRequestConverter::ConvertStreamRequest(const NCommon::TRequest& req) {
        TStreamRequest streamRequest{
            .StreamInfo = GetStreamInfo(req),
            .Aggregation = GetArg<ETimeAggregation>(req, ARG_AGGREGATION, false, ETimeAggregation::Day),
            .Offset = GetArg<size_t>(req, ARG_OFFSET, true),
            .Timestamp = TInstant::Seconds(GetArg<ui64>(req, ARG_CREATION_TIME, true)),
        };

        return streamRequest;
    }
}
