#pragma once

#include <util/datetime/base.h>
#include <util/generic/array_ref.h>
#include <util/generic/string.h>
#include <util/string/cast.h>

#include <optional>
#include <vector>

namespace NPassport::NLogstoreApi {
    enum class ETimeAggregation {
        Day,
        Hour,
    };

    using TStreamId = TStringBuf;

    enum class ECompressionCodec {
        PlainText = 0,
        GZip = 1,
        ZStd = 2,
    };

    struct TStreamInfo {
        TStringBuf File;
        TStringBuf Host;
        TStringBuf Env;
        TStreamId StreamId;

        TString LogableId() const;
    };

    struct TPushRequest {
        TStreamInfo StreamInfo;

        size_t Offset = 0;
        TString Body;
    };

    struct TStreamRequest {
        TStreamInfo StreamInfo;
        ETimeAggregation Aggregation = ETimeAggregation::Day;

        size_t Offset = 0;
        TInstant Timestamp;
    };

    struct TLogLine: TMoveOnly {
        TInstant Timestamp;
        TString Line;

        size_t Offset = 0;

        bool operator<(const TLogLine& other) const {
            return Offset < other.Offset;
        }

        bool operator==(const TLogLine& other) const {
            return Timestamp == other.Timestamp &&
                   Line == other.Line &&
                   Offset == other.Offset;
        }
    };

    // TODO: there should probably be a light-weight version of chunk that owns no data
    // We could then store a fully fledged version in the queue, which would be created from
    // the light-weight version, by finally copying all the data. This way we could avoid any copying
    // in the event that a chunk is written to file immediately
    class TChunk: TMoveOnly {
    public:
        using TContainer = TArrayRef<TLogLine>;

        TChunk(const TStreamInfo& streamInfo, std::vector<TLogLine>&& chunk)
            : FileId_(streamInfo.File)
            , Host_(streamInfo.Host)
            , Env_(streamInfo.Env)
            , StreamId_(streamInfo.StreamId)
            , Data_(std::move(chunk))
            , Lines_(Data_)
        {
            std::sort(Lines_.begin(), Lines_.end());
        }

        bool empty() const {
            return Lines_.empty();
        }

        bool Slice(size_t startOffset) {
            size_t i = 0;
            while (i < Data_.size() && Data_[i].Offset < startOffset) {
                ++i;
            }
            if (i >= Data_.size() || Data_[i].Offset != startOffset) {
                return false;
            }
            Lines_ = TArrayRef<TLogLine>(Data_).Slice(i);
            return true;
        }

        const TContainer& Lines() const {
            return Lines_;
        }

        ui64 Offset() const {
            Y_ENSURE(!Lines_.empty(), "Chunk is empty");

            return Lines_.begin()->Offset;
        }

        ui64 Roffset() const {
            Y_ENSURE(!Lines_.empty(), "Chunk is empty");

            return Lines_.rbegin()->Offset + Lines_.rbegin()->Line.size();
        }

        ui64 size() const {
            return Roffset() - Offset();
        }

        const TString& Host() const {
            return Host_;
        }

        const TString& File() const {
            return FileId_;
        }

        const TString& Env() const {
            return Env_;
        }

        const TString& Stream() const {
            return StreamId_;
        }

        void SetError(TString&& msg) {
            Error_ = std::move(msg);
        }

        const std::optional<TString>& Error() const {
            return Error_;
        }

        bool operator==(const TChunk& other) const {
            // TODO: check checksum?
            // Checks that the chunk is has the same source, offset and size
            return FileId_ == other.FileId_ &&
                   Host_ == other.Host_ &&
                   Env_ == other.Env_ &&
                   Offset() == other.Offset() &&
                   Roffset() == other.Roffset();
        }

    private:
        TString FileId_;
        TString Host_;
        TString Env_;
        TString StreamId_;

        std::vector<TLogLine> Data_;
        TArrayRef<TLogLine> Lines_;

        std::optional<TString> Error_;
    };
}

template <>
bool TryFromString<NPassport::NLogstoreApi::ETimeAggregation>(const TString& s, NPassport::NLogstoreApi::ETimeAggregation& result);
