#include "log_aggregator.h"

#include "logs_manager.h"
#include "simple_writer.h"

#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NLogstoreApi {
    TLogAggregator::TLogAggregator(std::unique_ptr<TLogsManager> logsManager,
                                   TLogAggregatorSettings&& settings)
        : Settings_(std::move(settings))
        , LogsManager_(std::move(logsManager))
        , ResourceDispatcher_(std::make_shared<NLb::TResourceDispatcher>("sequencer.enqueued_bytes"))
    {
        ResourceDispatcher_->AddResource(TSequencer::RESOURCE_ID, Settings_.MemoryLimit);
    }

    TLogAggregator::~TLogAggregator() = default;

    NThreading::TFuture<void> TLogAggregator::Aggregate(std::unique_ptr<TChunk> chunk) {
        Y_ENSURE(chunk, "Empty chunk container received");

        std::shared_ptr<TStream> stream = GetStream(chunk->Host(), chunk->File());
        if (stream->StreamId != chunk->Stream()) {
            TLog::Debug() << "LogAggregator " << stream->Sequencer.GetLogableId()
                          << ". Inode changed, please, reinitialize stream. In request: " << chunk->Stream()
                          << ". In current stream: " << stream->StreamId;
            throw TBadRequestError() << "Inode changed, please, reinitialize stream";
        }

        std::shared_ptr<IWriter> writer = LogsManager_->GetLogWriter(
            chunk->Env(),
            chunk->File(),
            stream->Aggregation);
        return stream->Sequencer.Write(*writer, std::move(chunk));
    }

    void TLogAggregator::AddStream(const TStreamRequest& request) {
        std::unique_lock lock(StreamLock_);

        TPerFile& perFile = Queue_[request.StreamInfo.File];
        if (!perFile.Unistat) {
            perFile.Unistat = std::make_shared<TSequencerUnistatCtx>(request.StreamInfo.File);
        }

        std::shared_ptr<TStream>& stream = perFile.StreamByHost[request.StreamInfo.Host];
        if (!stream) {
            stream = std::make_shared<TStream>(
                TSequencerSettings{
                    .LogableId = request.StreamInfo.LogableId(),
                    .BufferSize = Settings_.BufferLimit,
                },
                ResourceDispatcher_,
                perFile.Unistat);
        }

        // TODO: try to find a way to reliably keep track of creation time. For now this should do
        if (stream->Timestamp > request.Timestamp) {
            TLog::Debug() << "LogAggregator " << request.StreamInfo.LogableId()
                          << ". Got too old file. Time in request: " << request.Timestamp
                          << ". Time in current stream: " << stream->Timestamp;
            throw TBadRequestError() << "File is too old";
        }

        const EMissingChunkPolicy policy = stream->StreamId == request.StreamInfo.StreamId
                                               ? EMissingChunkPolicy::Discard
                                               : EMissingChunkPolicy::ForceFlush;

        {
            std::shared_ptr<IWriter> writer = LogsManager_->GetLogWriter(
                request.StreamInfo.Env,
                request.StreamInfo.File,
                request.Aggregation);
            stream->Sequencer.Reset(*writer, request.Offset, policy);
        }

        stream->StreamId = request.StreamInfo.StreamId;
        stream->Timestamp = request.Timestamp;
        stream->Aggregation = request.Aggregation;

        TLog::Debug() << "Session initialized: " << request.StreamInfo.LogableId()
                      << " with offset: " << request.Offset
                      << ", inode: " << stream->StreamId
                      << ", ts: " << stream->Timestamp
                      << ", aggregation: " << stream->Aggregation;
    }

    std::unique_ptr<TLogAggregator> TLogAggregator::Create(const NJson::TConfig& config, const TString& point) {
        TLogAggregatorSettings settings{
            .BufferLimit = config.As<size_t>(point + "/aggregator/buffer_limit__KB") * (1 << 10),
            .MemoryLimit = config.As<size_t>(point + "/aggregator/memory_limit__MB") * (1 << 20),
        };

        return std::make_unique<TLogAggregator>(
            TLogsManager::Create(config, point),
            std::move(settings));
    }

    void TLogAggregator::AddUnistat(NUnistat::TBuilder& builder) {
        LogsManager_->AddUnistat(builder);
        ResourceDispatcher_->AddUnistat(builder);
    }

    void TLogAggregator::AddUnistatForLogs(NUnistat::TBuilder& builder) {
        std::shared_lock lock(StreamLock_);
        for (auto& [file, perFile] : Queue_) {
            builder.Add(perFile.Unistat->TotalBytes);
            builder.Add(perFile.Unistat->TotalLines);
            builder.Add(perFile.Unistat->NonContiguous);
        }
    }

    std::shared_ptr<TStream> TLogAggregator::GetStream(const TStringBuf host, const TStringBuf file) {
        std::shared_lock lock(StreamLock_);

        auto itFile = Queue_.find(file);
        if (itFile == Queue_.end()) {
            throw TBadRequestError() << "No stream for " << host << ":" << file;
        }

        auto itHost = itFile->second.StreamByHost.find(host);
        if (itHost == itFile->second.StreamByHost.end()) {
            throw TBadRequestError() << "No stream for " << host << ":" << file;
        }

        return itHost->second;
    }
}
