#include "logs_manager.h"

#include "simple_writer.h"

#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>

namespace NPassport::NLogstoreApi {
    TLogsManager::TLogsManager(TFsPath storageDir,
                               TWriterFactoryPtr writerFactory,
                               std::vector<TString> envs)
        : StorageDir_(storageDir)
        , WriterFactory_(std::move(writerFactory))
    {
        for (const TString& env : envs) {
            Envs_.emplace(env, std::make_unique<TWritersMap>());
        }
    }

    TWriterPtr TLogsManager::GetLogWriter(TStringBuf env,
                                          TStringBuf file,
                                          ETimeAggregation aggregation,
                                          TInstant now) {
        TWritersMap& writers = GetWritersByEnv(env);

        {
            std::shared_lock lock(writers.Mutex);

            auto it = writers.Map.find(file);
            if (it != writers.Map.end()) {
                auto& wrapper = it->second;

                if (wrapper.Aggregation != aggregation) {
                    TLog::Debug() << "Aggregation changed for file '" << file << "': "
                                  << wrapper.Aggregation << " -> " << aggregation
                                  << ", applying after " << wrapper.Deadline
                                  << ". env=" << env;
                    wrapper.Aggregation = aggregation;
                }

                if (!wrapper.IsDeadlineReached(now)) {
                    return wrapper.Writer;
                }
            }
        }

        std::unique_lock lock(writers.Mutex);

        auto [it, inserted] = writers.Map.try_emplace(file, TWriterWrapper());
        if (inserted || it->second.IsDeadlineReached(now)) {
            if (it->second.Writer) {
                ToCleanup_.Enqueue(std::move(it->second.Writer));
            }

            it->second = CreateWriter(env, file, aggregation, now);
            TLog::Debug() << "New writer created: " << it->second.Filename
                          << " for env=" << env;
        }

        return it->second.Writer;
    }

    std::unique_ptr<TLogsManager> TLogsManager::Create(const NJson::TConfig& config, const TString& point) {
        std::vector<TString> envs = config.As<std::vector<TString>>(point + "/logs_manager/envs");
        Y_ENSURE(!envs.empty());

        return NUtils::TRegularTaskDecorator<TLogsManager>::CreateUnique(
            {
                .Period = TDuration::Seconds(config.As<ui64>(point + "/logs_manager/cleanup_period__sec")),
                .Name = "logs_manager",
            },
            config.As<TString>(point + "/logs_manager/storage_path"),
            std::make_shared<TSimpleWriterFactory>(),
            std::move(envs));
    }

    void TLogsManager::AddUnistat(NUnistat::TBuilder& builder) {
        for (auto& [env, writers] : Envs_) {
            builder.AddRow(NUtils::CreateStr("logs_manager.writers_", env), writers->Map.size());

            std::shared_lock lock(writers->Mutex);
            for (auto& [file, writer] : writers->Map) {
                writer.Writer->AddUnistat(builder);
            }
        }
    }

    void TLogsManager::Run() {
        Cleanup();
    }

    size_t TLogsManager::Cleanup(TInstant now) {
        // Destruction of writer can be long operation
        //  so we'll call dtor in background - without mutex protection
        std::vector<TWriterPtr> toDelete;
        ToCleanup_.DequeueAll(&toDelete);

        for (auto& [env, writers] : Envs_) {
            std::unique_lock lock(writers->Mutex);
            auto& map = writers->Map;

            for (auto it = map.begin(); it != map.end();) {
                if (it->second.IsDeadlineReached(now)) {
                    toDelete.push_back(std::move(it->second.Writer));
                    map.erase(it++);
                } else {
                    ++it;
                }
            }
        }

        return toDelete.size();
    }

    TLogsManager::TWritersMap& TLogsManager::GetWritersByEnv(TStringBuf env) {
        auto itEnv = Envs_.find(env);
        if (itEnv == Envs_.end()) {
            throw TBadRequestError() << "Env is unknown: '" << env << "'";
        }

        return *itEnv->second;
    }

    TLogsManager::TWriterWrapper TLogsManager::CreateWriter(TStringBuf env,
                                                            TStringBuf file,
                                                            ETimeAggregation aggregation,
                                                            TInstant now) {
        const TTimeAggregatorPtr aggregator = ITimeAggregator::Create(aggregation);

        const TString filename = StorageDir_ / env / NUtils::CreateStr(file, ".", aggregator->GetSuffix(now));

        return TWriterWrapper{
            .Writer = WriterFactory_->CreateWriter(filename),
            .Deadline = aggregator->GetDeadline(now),
            .Aggregation = aggregation,
            .Filename = filename,
        };
    }

    bool TLogsManager::TWriterWrapper::IsDeadlineReached(TInstant now) const {
        return Deadline <= now;
    }
}
