#pragma once

#include "base_writer.h"
#include "time_aggregator.h"

#include <util/datetime/base.h>
#include <util/folder/path.h>
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/thread/lfqueue.h>

#include <mutex>
#include <shared_mutex>
#include <unordered_set>

namespace NPassport::NJson {
    class TConfig;
}

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NLogstoreApi {
    class TLogsManager {
    public:
        TLogsManager(TFsPath storageDir,
                     TWriterFactoryPtr writerFactory,
                     std::vector<TString> envs = {"testing"});
        virtual ~TLogsManager() = default;

        TWriterPtr GetLogWriter(TStringBuf env,
                                TStringBuf file,
                                ETimeAggregation aggregation,
                                TInstant now = TInstant::Now());

        static std::unique_ptr<TLogsManager> Create(const NJson::TConfig& config, const TString& point);

        void AddUnistat(NUnistat::TBuilder& builder);

    public:
        struct TWriterWrapper {
            TWriterPtr Writer;
            TInstant Deadline;
            ETimeAggregation Aggregation = ETimeAggregation::Day;
            TString Filename;

            bool IsDeadlineReached(TInstant now) const;
        };

        struct TWritersMap {
            THashMap<TString, TWriterWrapper> Map;
            mutable std::shared_mutex Mutex;
        };

    protected:
        void Run();
        size_t Cleanup(TInstant now = TInstant::Now());

    private:
        TWritersMap& GetWritersByEnv(TStringBuf env);
        TWriterWrapper CreateWriter(TStringBuf env,
                                    TStringBuf file,
                                    ETimeAggregation aggregation,
                                    TInstant now);

    private:
        THashMap<TString, std::unique_ptr<TWritersMap>> Envs_;
        const TFsPath StorageDir_;
        const TWriterFactoryPtr WriterFactory_;

        TLockFreeQueue<TWriterPtr> ToCleanup_;
    };

    using TLogsManagerPtr = std::shared_ptr<TLogsManager>;
}
