#include "sequencer.h"

#include <passport/infra/daemons/logstoreapi/src/utils/exceptions.h>

#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/unistat/name_factory.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/string/subst.h>

namespace NPassport::NLogstoreApi {
    const TString TSequencer::RESOURCE_ID = "sequencer_queue";

    TSequencerUnistatCtx::TSequencerUnistatCtx(TStringBuf filename)
        : TotalBytes(NUtils::CreateStr("src_file=", filename, ";sequencer.total_bytes"))
        , TotalLines(NUtils::CreateStr("src_file=", filename, ";sequencer.total_lines"))
        , NonContiguous(NUtils::CreateStr("src_file=", filename, ";sequencer.non_contiguous_lines"))
    {
    }

    TSequencer::TSequencer(TSequencerSettings&& settings,
                           std::shared_ptr<NLb::TResourceDispatcher> resourceDispatcher,
                           TSequencerUnistatPtr unistat)
        : Settings_(std::move(settings))
        , ResourceDispatcher_(std::move(resourceDispatcher))
        , Unistat_(std::move(unistat))
        , BufferSize_(Settings_.BufferSize)
    {
        Y_ENSURE(Unistat_);
        TLog::Debug() << TLogCtx{this} << " New Sequencer was created";
    }

    TSequencer::~TSequencer() {
        TLog::Debug() << TLogCtx{this} << " Sequencer was desctroyed";
    }

    NThreading::TFuture<void> TSequencer::Write(IWriter& writer, std::unique_ptr<TChunk> chunk) {
        std::scoped_lock lock(Mutex_);

        if (ExpectedOffset_ + BufferSize_ < chunk->Offset()) {
            TLog::Debug() << TLogCtx{this} << chunk << ". chunk arrived too early; buffer_limit: " << BufferSize_;
            throw TBufferOverflowError() << "Chunk arrived too early";
        }

        if (ExpectedOffset_ > chunk->Offset()) {
            if (ExpectedOffset_ >= chunk->Roffset()) {
                TLog::Debug() << TLogCtx{this} << chunk << ". chunk arrived too late";
                throw TDuplicateError(ExpectedOffset_) << "Chunk arrived too late";
            }

            if (!chunk->Slice(ExpectedOffset_)) {
                TLog::Error() << TLogCtx{this} << chunk << ". chunk is misaligned";
                if (WritingStarted_) {
                    throw TBadRequestError() << "Log lines appear to be misaligned";
                }
                chunk->SetError("Corrupted stream: log lines appear to be misaligned");
            }
        }

        if (ExpectedOffset_ >= chunk->Offset()) {
            WritingStarted_ = true;

            WriteImpl(writer, std::move(chunk));
            FlushBuffer(writer);
            return NThreading::MakeFuture();
        }

        auto it = Buffer_.find(chunk);
        if (it != Buffer_.end()) {
            return it->second.GetFuture();
        }

        NThreading::TPromise<void> promise = NThreading::NewPromise<void>();
        NThreading::TFuture<void> future = promise.GetFuture();

        Push(std::move(chunk), std::move(promise));
        return future;
    }

    void TSequencer::Reset(IWriter& writer, size_t offset, EMissingChunkPolicy policy) {
        std::scoped_lock lock(Mutex_);

        TLog::Debug() << TLogCtx{this}
                      << "Trying to reset Sequenser: " << policy
                      << " mode with offset=" << offset
                      << ". Buffer size is " << Buffer_.size();

        switch (policy) {
            case EMissingChunkPolicy::Discard:
                if (ExpectedOffset_ < offset) {
                    TLog::Error() << TLogCtx{this}
                                  << ". Cannot discard buffer, chunks are missing. New offset is " << offset;
                    throw TMissingDataError(ExpectedOffset_) << "Some chunks are missing";
                }
                Clear();
                break;
            case EMissingChunkPolicy::ForceFlush:
                FlushBuffer(writer);
                if (!Buffer_.empty()) {
                    TLog::Warning() << TLogCtx{this} << "Force flushing " << Buffer_.size() << " chunks";
                    FlushBuffer(writer, true);
                }
                ExpectedOffset_ = offset;
                break;
        }

        WritingStarted_ = false;

        TLog::Debug() << TLogCtx{this} << ". Sequenser was reseted: buffer size is " << Buffer_.size();
    }

    const TString& TSequencer::GetLogableId() const {
        return Settings_.LogableId;
    }

    void TSequencer::Clear() {
        size_t size = 0;
        for (const auto& [key, _] : Buffer_) {
            size += key->size();
        }
        if (!ResourceDispatcher_->Release(RESOURCE_ID, size)) {
            TLog::Error() << TLogCtx{this} << "Trying to release more memory than allocated";
        }
        Buffer_.clear();
    }

    TChunkValue TSequencer::Pop() {
        auto handle = Buffer_.extract(Buffer_.begin());
        if (!ResourceDispatcher_->Release(RESOURCE_ID, handle.key()->size())) {
            TLog::Error() << TLogCtx{this} << "Trying to release more memory than allocated";
        }
        return std::make_pair(std::move(handle.key()), std::move(handle.mapped()));
    }

    void TSequencer::Push(std::unique_ptr<TChunk> chunk, NThreading::TPromise<void>&& promise) {
        if (!ResourceDispatcher_->TryAcquire(RESOURCE_ID, chunk->size())) {
            TLog::Warning() << TLogCtx{this} << chunk << "Memory limit reached, please try backing off";
            throw TBufferOverflowError() << "Memory limit reached, please try backing off";
        }
        Buffer_.emplace(std::move(chunk), std::move(promise));
    }

    void TSequencer::FlushBuffer(IWriter& writer, bool force) {
        while (!Buffer_.empty()) {
            if (!Contiguous()) {
                if (!force && !Overlapping()) {
                    break;
                }

                ++Unistat_->NonContiguous;
                TLog::Error() << TLogCtx{this} << Buffer_.begin()->first << "buffer is non-contiguous";
            }

            auto [chunk, future] = Pop();
            WriteImpl(writer, std::move(chunk));
            future.SetValue();
        }
    }

    void TSequencer::WriteImpl(IWriter& writer, std::unique_ptr<TChunk> chunk) {
        writer.Write(*chunk).Wait();
        ExpectedOffset_ = chunk->Roffset();

        Unistat_->TotalBytes += chunk->size();
        Unistat_->TotalLines += chunk->Lines().size();
    }

    bool TSequencer::Contiguous() const {
        return !Buffer_.empty() && (ExpectedOffset_ == Buffer_.begin()->first->Offset());
    }

    bool TSequencer::Overlapping() const {
        return !Buffer_.empty() && (ExpectedOffset_ > Buffer_.begin()->first->Offset());
    }
}

using namespace NPassport::NLogstoreApi;

template <>
void Out<std::unique_ptr<TChunk>>(IOutputStream& o, const std::unique_ptr<TChunk>& value) {
    o << "(got chunk: " << value->Offset() << "->" << value->Roffset() << ")";
}

template <>
void Out<TSequencer::TLogCtx>(IOutputStream& o, const TSequencer::TLogCtx& value) {
    o << "Sequencer[" << value.Settings().LogableId
      << "];expected_offset=" << value.ExpectedOffset() << ".";
}
