#pragma once

#include "base_writer.h"

#include <passport/infra/daemons/logstoreapi/src/utils/data_types.h>

#include <passport/infra/libs/cpp/unistat/diff.h>

#include <library/cpp/threading/future/future.h>

#include <map>
#include <mutex>

namespace NPassport {
    class TLoggingStream;
}

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NLb {
    class TResourceDispatcher;
}

namespace NPassport::NLogstoreApi {
    struct TChunkPtrComparator {
        bool operator()(const std::unique_ptr<TChunk>& lhs, const std::unique_ptr<TChunk>& rhs) const {
            return lhs->Offset() < rhs->Offset();
        }
    };

    using TChunkBuffer = std::map<std::unique_ptr<TChunk>, NThreading::TPromise<void>, TChunkPtrComparator>;
    using TChunkValue = std::pair<std::unique_ptr<TChunk>, NThreading::TPromise<void>>;

    enum class EMissingChunkPolicy {
        Discard,
        ForceFlush,
    };

    struct TSequencerUnistatCtx {
        TSequencerUnistatCtx(TStringBuf filename);

        NUnistat::TSignalDiff<ui64> TotalBytes;
        NUnistat::TSignalDiff<ui64> TotalLines;
        NUnistat::TSignalDiff<ui64> NonContiguous;
    };
    using TSequencerUnistatPtr = std::shared_ptr<TSequencerUnistatCtx>;

    struct TSequencerSettings {
        TString LogableId;

        size_t BufferSize = 4096;
    };

    class TSequencer {
    public:
        TSequencer(TSequencerSettings&& settings,
                   std::shared_ptr<NLb::TResourceDispatcher> resourceDispatcher,
                   TSequencerUnistatPtr unistat);
        ~TSequencer();

        NThreading::TFuture<void> Write(IWriter& writer, std::unique_ptr<TChunk> chunk);

        void Reset(IWriter& writer, size_t offset, EMissingChunkPolicy policy);

        const TString& GetLogableId() const;

    public:
        struct TLogCtx {
            TSequencer* This = nullptr;

            const TSequencerSettings& Settings() const {
                return This->Settings_;
            }

            size_t ExpectedOffset() const {
                return This->ExpectedOffset_;
            }
        };

        static const TString RESOURCE_ID;

    protected:
        void Push(std::unique_ptr<TChunk> chunk, NThreading::TPromise<void>&& promise);

        void FlushBuffer(IWriter& writer, bool force = false);

        void WriteImpl(IWriter& writer, std::unique_ptr<TChunk> chunk);

        bool Contiguous() const;
        bool Overlapping() const;

        void Clear();
        TChunkValue Pop();

    private:
        const TSequencerSettings Settings_;

        std::shared_ptr<NLb::TResourceDispatcher> ResourceDispatcher_;
        TSequencerUnistatPtr Unistat_;

        const size_t BufferSize_;
        TChunkBuffer Buffer_;
        std::mutex Mutex_;

        bool WritingStarted_ = false;
        size_t ExpectedOffset_ = 0;
    };
}
